#! /usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import print_function

import requests
import json
import os
import sys


def read_uids(uids_file):
    uids = set()
    with open(uids_file) as f:
        for line in f:
            uid = line.strip()
            uids.add(uid)
    return uids


def get_user_meetings(uid, date_from, date_to):
    url = "http://calendar-api.tools.yandex.net/internal/get-events.json"
    url += "?uid={}&from={}&to={}".format(uid, date_from, date_to)
    tries = 0
    while True:
        try:
            resp = requests.get(url, timeout=30)
            if resp.status_code != 200:
                raise Exception("calendar response error: {} {}".format(resp.status_code, resp.text))
            break
        except:
            tries += 1
            if tries == 5:
                raise
    return json.loads(resp.text)


def get_existing_uids(events_dir):
    return os.listdir(events_dir)


def get_new_uids(uids, existing_uids):
    return set(uids) - set(existing_uids)


def dump_events(events_dir, uids, periods):
    for i, uid in enumerate(uids):
        events = []
        for date_from, date_to in periods:
            user_meetings = get_user_meetings(uid, date_from, date_to)
            for meeting in user_meetings.get("events", []):
                event_datetime = meeting['startTs'].split('T')
                event_date = event_datetime[0]
                event_time = event_datetime[1]
                try:
                    organizer = meeting["organizer"]["login"]
                except KeyError:
                    organizer = None
                rooms_data = []
                for room in meeting["resources"]:
                    if room["resourceType"] != "room":
                        continue
                    rooms_data.append({
                        "name": room["name"],
                        "resourceType": room["resourceType"],
                        "description": room["description"],
                    })
                event = {
                    "availability": meeting.get('availability'),
                    "event_id": meeting["id"],
                    "name": meeting["name"],
                    "organizer": organizer,
                    "rooms": rooms_data,
                    "date": event_date,
                    "time": event_time,
                    "startTs": meeting['startTs'],
                    "endTs": meeting['endTs'],
                }

                is_closed_event = meeting.get('primaryLayerClosed') or not meeting.get('othersCanView', True)
                if is_closed_event:
                    for private_info_key in [
                        'event_id',
                        'name',
                        'organizer',
                        'rooms',
                    ]:
                        event[private_info_key] = None
                events.append(event)

        with open('{}/{}'.format(events_dir, uid), 'w') as output_file:
            output_file.write(json.dumps(events, indent=4))
        print("dumped {} events for {} (#{})".format(len(events), uid, i))


def main():
    if len(sys.argv) < 3:
        print("usage: {} <staff uids file> <events dir>".format(sys.argv[0]))
        return 1
    staff_uids_file = sys.argv[1]
    events_dir = sys.argv[2]
    uids = read_uids(staff_uids_file)
    #uids = [1120000000007292, 1120000000000321, 1120000000014351] # dskut, jkennedy, lyadzhin
    print("read {} uids".format(len(uids)))
    periods = [
        ("2018-09-01", "2018-10-01"),
        ("2018-10-01", "2018-11-01"),
        ("2018-11-01", "2018-12-01"),
    ]

    existing_uids = get_existing_uids(events_dir)
    print("already dumped events for {} uids".format(len(existing_uids)))
    new_uids = get_new_uids(uids, existing_uids)
    print("will dump for {} new uids".format(len(new_uids)))
    dump_events(events_dir, new_uids, periods)
    print("done")


if __name__ == '__main__':
    main()
