import json
from dataclasses import dataclass
from datetime import date

import aiohttp
from aiohttp import ClientSession


@dataclass
class Event:
    id: int

    def __str__(self):
        return f"event:{self.id}"

    def __repr__(self):
        return str(self)


@dataclass
class DateRange:
    since: date  # inclusive
    until: date  # exclusive

    def __str__(self):
        return f"[{self.since}, {self.until})"

    def __repr__(self):
        return str(self)


class Api:
    def __init__(self):
        self.base_url = "http://calendar-api.calendar-prestable.yandex-team.ru/internal"

    def __to_event(self, event: dict) -> Event:
        return Event(id=int(event['id']))

    async def __get(self, session: ClientSession, url: str) -> dict:
        async with session.get(url) as response:
            if response.status != 200:
                raise IOError(f"api request failed: {response.text()}")
            return json.loads(await response.text())

    def __is_subscription(self, event: dict) -> bool:
        return 'decision' not in event

    def __is_invisible_event(self, event: dict) -> bool:
        return 'name' not in event

    async def get_events(self, uid: int, date_range: DateRange) -> [Event]:
        async with aiohttp.ClientSession() as session:
            response = await self.__get(session, f"{self.base_url}/get-events?uid={uid}"
                                                 f"&limitAttendees=true&hideAbsences=true"
                                                 f"&from={date_range.since}&to={date_range.until}&showDeclined=true"
                                                 f"&fields=events/id,events/decision,events")
            return [self.__to_event(event) for event in response['events'] if not self.__is_subscription(event)]

    async def select_visible_events(self, event_ids: [int], for_uid: int) -> [int]:
        ids_string = ','.join([str(id) for id in event_ids])
        async with aiohttp.ClientSession() as session:
            response = await self.__get(session, f"{self.base_url}/get-events-brief?uid={for_uid}&forResource=false"
                                                 f"&eventIds={ids_string}")
            return [event['id'] for event in response['events'] if not self.__is_invisible_event(event)]
