import argparse
import asyncio
from dataclasses import dataclass
from datetime import timedelta, date, datetime
from typing import Iterable

from Api import Api, DateRange
from dao import Dao


@dataclass
class Settings:
    host: str
    port: int
    user: str
    dbname: str
    password: str
    timeout_sec: int
    dry_run: bool
    secret_user: int
    since: date
    until: date
    visibility_check_user: int


def iterate(date_range: DateRange, delta: timedelta) -> Iterable[DateRange]:
    def next_range(since: date) -> DateRange:
        return DateRange(since=since, until=min(since + delta, date_range.until))

    current = next_range(since=date_range.since)
    while current.since < date_range.until:
        yield current
        current = next_range(since=current.until)


async def close_secrets(settings: Settings) -> None:
    dao = Dao(host=settings.host, port=settings.port, user=settings.user, dbname=settings.dbname,
              dry_run=settings.dry_run, timeout_sec=settings.timeout_sec, password=settings.password)

    api = Api()

    await dao.connect(pool_size=10)
    await dao.ping()

    for sub_range in iterate(DateRange(settings.since, settings.until), delta=timedelta(days=30)):
        events = await api.get_events(uid=settings.secret_user, date_range=sub_range)
        print(f"range - {sub_range}:{events}")
        if len(events) > 0:
            visible_event_ids = await api.select_visible_events(event_ids=[event.id for event in events],
                                                                for_uid=settings.visibility_check_user)
            if len(visible_event_ids) > 0:
                events = [event for event in events if event.id not in visible_event_ids]
                visible_event_ids_list_str = ','.join([str(event_id) for event_id in visible_event_ids])
                print(f"skipped events: {visible_event_ids_list_str}")

            event_ids = [event.id for event in events]
            await dao.close_events(event_ids=event_ids)
            print("range closed")

            visible_event_ids = await api.select_visible_events(event_ids=event_ids, for_uid=settings.secret_user)
            if set(visible_event_ids) != set(event_ids):
                print("Alarm!!!! We close events from super secret!!!")
                raise RuntimeError("Failed")
        else:
            print("nothing to close")


def parse_date(date_str: str) -> date:
    return datetime.strptime(date_str, '%Y-%m-%d').date()


def parse_args() -> Settings:
    parser = argparse.ArgumentParser(prog="migrator", description="migrate closed layers events")
    parser.add_argument("--host", type=str, required=True, help="database host")
    parser.add_argument("--port", type=int, required=True, help="database port")
    parser.add_argument("--user", type=str, required=True, help="database username")
    parser.add_argument("--dbname", type=str, required=True, help="database name")
    parser.add_argument("--password", type=str, required=True, help="database password")
    parser.add_argument("--timeout_sec", type=int, required=True, help="query timeout")
    parser.add_argument("--secret_user", type=int, required=True, help="secret user uid")
    parser.add_argument("--since", type=str, required=True, help="start search date")
    parser.add_argument("--until", type=str, required=True, help="end search date")
    parser.add_argument("--visibility_check_user", type=int, required=True,
                        help="user uid without permission to see closing events")
    parser.add_argument("--dry_run", default=False, type=lambda x: (str(x).lower() == 'true'),
                        required=True, help="execute migration without data update")

    args = parser.parse_args()
    return Settings(host=args.host, port=args.port, user=args.user, dbname=args.dbname, password=args.password,
                    timeout_sec=args.timeout_sec, dry_run=args.dry_run, secret_user=args.secret_user,
                    since=parse_date(args.since), until=parse_date(args.until),
                    visibility_check_user=args.visibility_check_user)


async def main():
    settings = parse_args()
    print(f"start with settings: {settings}")
    await close_secrets(settings)


if __name__ == "__main__":
    asyncio.run(main())
