import argparse
import asyncio
from dataclasses import dataclass
from typing import Optional

from dao import Dao


@dataclass
class Settings:
    host: str
    port: int
    user: str
    dbname: str
    password: str
    timeout_sec: int
    dry_run: bool
    batch_size: int
    parallel: int
    delay: float
    start_id: Optional[int]


async def migrate_events(settings: Settings) -> None:
    dao = Dao(host=settings.host, port=settings.port, user=settings.user, dbname=settings.dbname,
              dry_run=settings.dry_run, timeout_sec=settings.timeout_sec, password=settings.password,
              delay=settings.delay)

    await dao.connect(pool_size=settings.parallel)
    await dao.ping()

    if settings.parallel == 0:
        start_id = settings.start_id
        await dao.migrate_all_events(batch_size=settings.batch_size, since_id=start_id if start_id is not None else -1)
    else:
        await dao.batch_migrate_events(batch_size=settings.batch_size, simultaneous_batches_count=settings.parallel)


def parse_args() -> Settings:
    parser = argparse.ArgumentParser(prog="migrator", description="migrate closed layers events")
    parser.add_argument("--host", type=str, required=True, help="database host")
    parser.add_argument("--port", type=int, required=True, help="database port")
    parser.add_argument("--user", type=str, required=True, help="database username")
    parser.add_argument("--dbname", type=str, required=True, help="database name")
    parser.add_argument("--password", type=str, required=True, help="database password")
    parser.add_argument("--timeout_sec", type=int, required=True, help="query timeout")
    parser.add_argument("--dry_run", type=str, required=True, help="execute migration without data update")
    parser.add_argument("--batch_size", type=int, required=True, help="migration batch size")
    parser.add_argument("--parallel", type=int, required=True, help="parallel queries count")
    parser.add_argument("--delay", type=float, required=True, help="delay between queries (seconds)")
    parser.add_argument("--start_id", type=int, required=False, help="start non-parallel migration since passed id")

    args = parser.parse_args()
    dry_run = False if args.dry_run == "false" else True
    print(f"dry run is {dry_run}")
    return Settings(host=args.host, port=args.port, user=args.user, dbname=args.dbname, password=args.password,
                    timeout_sec=args.timeout_sec, dry_run=dry_run, batch_size=args.batch_size,
                    parallel=args.parallel, delay=args.delay, start_id=args.start_id)


async def main():
    try:
        settings = parse_args()
        print(f"start with settings: {settings}")
        await migrate_events(settings)
    except Exception as ex:
        print(f"Execution failed: {ex}")


if __name__ == "__main__":
    asyncio.run(main())
