import argparse
import subprocess
import sys


class Component:
    def __init__(self, component_name, application_name):
        self.component_name = component_name
        self.application_name = application_name

    def prepare_command(self, action):
        return f'supervisorctl {action} {self.application_name}'

    def prepare_executer_command(self, dc, action):
        command_to_run = self.prepare_command(action)
        return ['executer',  '-c', 'p_exec', f'%{self.component_name}@{dc}', command_to_run]


class WebComponent(Component):
    def __init__(self, component_name):
        super().__init__(component_name, 'nginx')


class WebComponentWithStophook(WebComponent):
    def prepare_command(self, action):
        if action == 'stop':
            return f'curl "localhost:23456/" ; sleep 10; {super().prepare_command(action)}'
        return super().prepare_command(action)


class SharpeiComponent:
    def __init__(self, component_name):
        self.component_name = component_name

    def prepare_command(self, action):
        assert action in ('start', 'stop'), 'unknown action requested'
        if action == 'stop':
            return 'manage-balancer close'
        else:
            return 'manage-balancer open'

    def prepare_executer_command(self, dc, action):
        if dc not in ('iva', 'myt', 'sas', 'vla'):
            return ['echo', f'There are no hosts in {dc} for {self.component_name}']
        elif action == 'stop' or action == 'start':
            command_to_run = self.prepare_command(action)
            return ['executer',  '-c', 'p_exec', f'n:{self.component_name}_{dc}', command_to_run]
        else:
            return ['echo', f'"Unsupported command \'{action}\' for {self.__class__.__name__}"']


def parse_args():
    parser = argparse.ArgumentParser(prog='run_command_on_all_hosts', description='')
    parser.add_argument('action', choices=['start', 'stop', 'status'], help='supervisorctl action to run at all hosts')
    parser.add_argument('--dc', type=str, required=True, help='datacenter')
    parser.add_argument('--silent', action='store_true', help='do not ask approval before running the command at each hosts, only once at start')
    parser.add_argument('--cache', action='store_true', help="load info about Qloud hosts from cache (do not run 'executer' without '-c' preliminarily). If your executer cache is up-to-date it will speed up the process.")
    return parser.parse_args()


def user_approves():
    return input("Enter 'y' to proceed: ") == 'y'


def run_for(components):
    args = parse_args()
    print(args)
    if not user_approves():
        sys.exit()

    if not args.cache:  # update executer cache
        subprocess.check_call(['executer', '""'])

    for component in components:
        print('=' * 80)
        executer_command = component.prepare_executer_command(args.dc, args.action)
        print(f"Will run command: {' '.join(executer_command)!r}")
        if args.silent or user_approves():
            subprocess.check_call(executer_command)
