# coding=utf-8
import logging
from collections import defaultdict
from datetime import datetime

from startrek_client import Startrek

import charts_api
import mail_api
from alert_renderer import render_alerts
from report_render import render_report_items

RAW_SUBJECT_FORMAT = u"{subject} {from_ts} - {to_ts}"
logger = logging.getLogger('app')
DEADLINE_DATE = '%Y-%m-%d'


class LastCommentItem:
    def __init__(self, text, date):
        self.text = text
        self.date = date


class MetricItem:
    def __init__(self, text, url, cached_img_file):
        self.text = text
        self.url = url
        self.cached_img_file = cached_img_file


class ReportItem:
    def __init__(self, goals_id, assignee, title, status, importance, deadline, comment_items, metrics):
        self.id = goals_id
        self.assignee = assignee
        self.title = title
        self.status = status
        self.importance = importance
        self.deadline = deadline
        self.latestComments = comment_items
        self.metrics = metrics


class Report:
    def __init__(self, goals_filter_ids, from_ts, to_ts, sender,
                 subject, goals_token, charts_token, smtp_login, smtp_password, short_comments,
                 mode):
        self.goals_filter_ids = goals_filter_ids
        self.from_ts = from_ts
        self.to_ts = to_ts
        self.email = None
        self.report_items = []
        self.sender = sender
        self.subject = subject
        self.mode = mode
        self.short_comments = short_comments
        self.startrek_client = Startrek(useragent='goals-reporter', token=goals_token,
                                        base_url='https://st-api.yandex-team.ru')
        charts_api.set_token(charts_token)
        mail_api.set_credentials(smtp_login, smtp_password)

    def __format_subject(self, text):
        return RAW_SUBJECT_FORMAT.format(
            subject=text,
            from_ts=str(self.from_ts.date()),
            to_ts=str(self.to_ts.date())
        )

    @staticmethod
    def __format_deadline(deadline):
        if deadline is None:
            return u'No deadline'
        else:
            dead_date = datetime.strptime(deadline, DEADLINE_DATE)
            return str(dead_date.year) + ' Q' + str(((dead_date.month - 1) // 3) + 1)

    def __create(self):
        issues = []
        for goals_filter_id in self.goals_filter_ids:
            issues_find = self.startrek_client.issues.find(filter_id=goals_filter_id)
            logger.debug('found' + str(len(issues_find)) + ' issues on filter ' + str(goals_filter_id))
            for issue in issues_find:
                issues.append(issue)
        logger.debug('total found' + str(len(issues)))

        for issue in issues:
            task_json = issue.as_dict()
            metrics = list()
            if 'checklistItems' in task_json:
                for checklistItem in task_json['checklistItems']:
                    if checklistItem['checklistItemType'] == 'metric' and checklistItem.get('url') is not None:
                        metrics.append(MetricItem(checklistItem['text'], checklistItem['url'],
                                                  charts_api.get_metric_img(checklistItem['url'])))
            last_comments = list()
            issue_comments = list()
            for comment in issue.comments:
                issue_comments.append(comment)
            if len(issue_comments) > 0:
                last_comment = issue_comments[-1]
                comment_item = LastCommentItem(last_comment['text'], last_comment['createdAt'])
                if str(self.from_ts) <= last_comment['createdAt']:
                    last_comments.append(comment_item)
            report_item = ReportItem(
                task_json['key'][6:],
                task_json['assignee']['id'],
                task_json['summary'],
                task_json['status']['display'],
                task_json['goalImportance']['display'],
                self.__format_deadline(task_json.get('deadline', None)),
                last_comments,
                metrics
            )
            self.report_items.append(report_item)

    def __make_report(self):
        rendered = render_report_items(self.report_items, self.short_comments)
        self.email = mail_api.compose(rendered, self.report_items)

    @staticmethod
    def __get_login2report(report_items, report_filter):
        login2goals = defaultdict(list)
        for report_item in report_items:
            if report_filter(report_item):  # latestComments
                logger.info('user: %s, goals: %s', report_item.assignee, report_item.id)
                login2goals[report_item.assignee].append(report_item.id)
        return login2goals

    def send_alerts(self, receivers):
        if len(self.report_items) == 0:
            self.__create()

        login2report = self.__get_login2report(self.report_items, lambda x: len(x.latestComments) == 0)

        for login in login2report.keys():
            alert_receivers = list(receivers)
            alert_receivers.append(str(login + '@yandex-team.ru'))
            report_items_ids = login2report[login]
            alert_reports = list()
            for report_item in self.report_items:
                if report_item.id in report_items_ids:
                    alert_reports.append(report_item)
            subject = self.__format_subject(str(login + '@') + u' Нет комментариев в целях за')

            rendered_alert = render_alerts(alert_reports, self.short_comments)
            email_alert = mail_api.compose(rendered_alert, [])
            mail_api.send(email_alert, self.sender, alert_receivers, subject)

    def dump_report(self, report_file):
        if self.email is None:
            self.__create()
            self.__make_report()
        with open(report_file, 'w') as report_file:
            report_file.write(str(self.email))

    def send_report(self, receivers, goals_receivers):
        if self.email is None:
            self.__create()
            self.__make_report()
        report_receivers = set(receivers)
        if goals_receivers:
            login2goals = self.__get_login2report(self.report_items, lambda x: x)
            for login in login2goals.keys():
                report_receivers.add(str(login + '@yandex-team.ru'))

        mail_api.send(self.email, self.sender, report_receivers,
                      self.__format_subject(self.subject))
