#!/usr/bin/env python
# coding=utf-8
import argparse
import logging
from datetime import timedelta, datetime
from sys import argv

import log_config
from report import Report

INPUT_DATE_FORMAT = '%Y-%m-%d'

logger = logging.getLogger('app')


def main():
    log_config.setup()

    parser = argparse.ArgumentParser(description='Goals report mailer')

    parser.add_argument('-e', '--emails', action='store', dest='receivers', required=True,
                        help='Emails to send report to', nargs='+')
    parser.add_argument('-subj', '--subject', action='store', dest='subject', required=True, help='Emails subject')
    parser.add_argument('-gr', '--goals-receivers', action='store', dest='goals_receivers', required=False,
                        help='Send also to goals owners')

    parser.add_argument('-f', '--from', action='store', dest='from_ts', required=False,
                        help='Initial date of report in format YYYY-mm-dd, inclusive')
    parser.add_argument('-t', '--to', action='store', dest='to_ts', required=False,
                        help='Final date of report in format YYYY-mm-dd, exclusive')
    parser.add_argument('-p', '--period', action='store', dest='period', required=False,
                        help='')

    parser.add_argument('-gdid', '--goals_filter_ids', type=int, action='store', dest='goals_filter_ids', nargs='+',
                        required=False,
                        help='Specific goals ids tracker filterId')

    parser.add_argument('-lc', '--long_comments', action="store_true", help='Do not shrink comments if specified')

    parser.add_argument('-m', '--mode', action="store", required=False, help='mode')

    parser.add_argument('-at', '--token', action='store', dest='token',
                        help='Access token')
    parser.add_argument('-gt', '--goals_token', action='store', dest='goals_token',
                        help='Goals access token')
    parser.add_argument('-ct', '--charts_token', action='store', dest='charts_token',
                        help='Charts access token')

    parser.add_argument('-s', '--sender', action='store', dest='sender_address', required=True,
                        help='Email address to send email from')

    parser.add_argument('-sl', '--smtp_login', action='store', dest='smtp_login', required=False,
                        help='Login to send email via SMTP')
    parser.add_argument('-sp', '--smtp_pass', action='store', dest='smtp_pass', required=True,
                        help='Password to send email via SMTP')

    parser.add_argument('-dump', '--dump_report', action='store', dest='report_file', required=False,
                        help='File to write report to')

    arguments = parser.parse_args(argv[1:])

    smtp_password = arguments.smtp_pass
    smtp_login = arguments.smtp_login or arguments.sender_address

    short_comments = True
    if arguments.long_comments:
        short_comments = False

    goals_receivers = False
    if arguments.goals_receivers:
        goals_receivers = True

    mode = 'REPORT'
    if arguments.mode:
        mode = arguments.mode

    goals_token = arguments.goals_token or arguments.token
    if goals_token is None:
        raise AttributeError('No goals token provided')
    charts_token = arguments.charts_token or arguments.token
    if charts_token is None:
        raise AttributeError('No charts token provided')

    sender = arguments.sender_address
    receivers = arguments.receivers
    subject = arguments.subject

    period = arguments.period
    if period is not None:
        now = datetime.now()
        start_time = now.replace(hour=0, minute=0, second=0, microsecond=0)
        from_ts = (start_time - timedelta(days=int(period)))
        to_ts = (start_time + timedelta(days=1))
    else:
        from_ts = datetime.strptime(arguments.from_ts, INPUT_DATE_FORMAT)
        to_ts = datetime.strptime(arguments.to_ts, INPUT_DATE_FORMAT)

    goals_filter_ids = arguments.goals_filter_ids

    report = Report(
        goals_filter_ids=goals_filter_ids,
        from_ts=from_ts,
        to_ts=to_ts,
        sender=sender,
        subject=subject,
        goals_token=goals_token,
        charts_token=charts_token,
        smtp_login=smtp_login,
        smtp_password=smtp_password,
        short_comments=short_comments,
        mode=mode
    )

    if mode == 'ALERT':
        report.send_alerts(receivers)
    else:
        if arguments.report_file is not None:
            report.dump_report(arguments.report_file)

        report.send_report(receivers, goals_receivers)


if __name__ == '__main__':
    main()
