#! /bin/bash
set -exo pipefail

gzip=${gzip:-gzip}

function incremental
{
  local force=$1
  local -a rotated
  for log_path in "${logs[@]}"
  do
    if [ -f "${log_path}.tmp" ]
    then
      rotated+=("$log_path")
      continue
    fi
    if [ ! -f "${log_path}" ]
    then
        continue
    fi
    local log_size=$(stat -c %s "$log_path")
    # global maxsize
    if [[ $log_size -gt "$maxsize" || "$force" ]]
    then
      mv "$log_path" "${log_path}.tmp"
      rotated+=("$log_path")
    fi
  done
  if [[ $(type -t postrotate) == function && ${#rotated[@]} -gt 0 ]]
  then
    postrotate
  fi
  # global rate_limit freeing IO
  for log_path in "${rotated[@]}"
  do
    if [ "$rate_limit" ]
    then
      ${gzip} -c ${gzip_opts} < "${log_path}.tmp" | pv -L "$rate_limit" >> "${log_path}.i.gz"
    else
      ${gzip} -c ${gzip_opts} < "${log_path}.tmp" >> "${log_path}.i.gz"
    fi
    rm "${log_path}.tmp"
  done

  # global drop_old
  if [ "$drop_old" ]
  then
    size=$(df -h /var/log | awk '$(NF-1) ~ /\%$/ && $1 != "Filesystem" { print $(NF-1)}' | sed 's/%//')
    if [[ $size =~ [0-9]+ && $size -ge 95 ]]
      then
      for log_path in "${logs[@]}"
      do
        old_one=$(ls -1rt $log_path* | head -1)
        if [[ ! $old_one =~ [0i].gz ]]
        then
          rm "$old_one"
        fi
      done
    fi
  fi
}


function full
{
  incremental force
  for log_path in ${logs[@]}
  do
    # global rotate
    local x=$((rotate - 1))  # last index
    while [[ "$x" -gt 0 ]]
    do
      if [ -e "${log_path}.$((x-1)).gz" ]
      then
        mv "${log_path}.$((x-1)).gz" "${log_path}.${x}.gz"
      fi
      ((x--))
    done
    if [ -f "${log_path}.i.gz" ]
    then
      mv "${log_path}.i.gz" "${log_path}.0.gz"
    fi
  done
}


function run_all
{
  local mode=$1
  shopt -s nullglob
  for c in $configs
  do
    if [ $parallel -gt 0 ]
    then
        eval "$0" "$mode" "$c" &
    else
        eval "$0" "$mode" "$c"
    fi
  done
  wait
}


mode=$1
config=$2
configs=${configs:-/etc/logrotate-incremental.d/*.conf}
parallel=${parallel:-1}

if [[ ! "$mode" =~ --incremental|--full ]]
then
  echo "Usage: $0 [--incremental|--full] [<log_path>]" >&2
  exit 1
fi

[[ "$mode" == --incremental ]] && nonblock="-n"
[[ "${LOCKED}" != "$0" ]] && LOCKED="$0" exec flock $nonblock "$0" "$0" "$@"

if [[ "$config" == '' ]]
then
  run_all "$mode"
else
  source "$config"
  maxsize=${maxsize:-$((1024**3))}
  rotate=${rotate:-5}
  cmd=${mode#--}
  eval $cmd
fi
