package ru.yandex.olelole;

import java.io.File;
import java.nio.charset.StandardCharsets;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.http.HttpHost;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

import ru.yandex.http.util.CharsetUtils;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.olelole.config.AlertsConfigBuilder;
import ru.yandex.olelole.config.ImmutableAlertConfig;
import ru.yandex.olelole.config.ImmutableAlertsConfig;
import ru.yandex.olelole.config.ImmutableNotificationConfig;
import ru.yandex.olelole.config.NotificationType;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.util.string.StringUtils;

public class AlertsUploader {
    private static final String ALERTS_PREFIX = "ps.gen.";
    private static final HttpHost API_HOST =
        new HttpHost("yasm.yandex-team.ru", 443, "https");
    private final CloseableHttpClient yasmClient = HttpClients.createDefault();

    private void upload(
        final ImmutableAlertsConfig config,
        final boolean justPrint)
        throws Exception
    {
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
            writer.startObject();
            writer.key("prefix");
            writer.value(buildAlertPrefix(config));
            writer.key("alerts");
            writer.startArray();
            for (ImmutableAlertConfig alertConfig: config.alerts()) {
                writeAlert(writer, alertConfig, config);
            }
            writer.endArray();
            writer.endObject();
        }

        if (!justPrint) {
            HttpPost post = new HttpPost("/srvambry/alerts/replace");
            post.setEntity(new StringEntity(sbw.toString(), StandardCharsets.UTF_8));
            CloseableHttpResponse response = yasmClient.execute(API_HOST, post);
            System.out.println(CharsetUtils.toString(response.getEntity()));
        } else {
            System.out.println(sbw.toString());
        }
    }

    private void writeAlert(
        final JsonWriter writer,
        final ImmutableAlertConfig alert,
        final ImmutableAlertsConfig config)
        throws Exception
    {
        String alertName = buildAlertPrefix(config) + '.' + alert.name();
        writer.startObject();
        writer.key("name");
        writer.value(alertName);
        writer.key("disaster");
        writer.value(false);
        writer.key("abc");
        writer.value(config.abc());
        writer.key("signal");
        writer.value(alert.signal());
        writer.key("mgroups");
        writer.value(Collections.singletonList("ASEARCH"));
        writer.key("tags");
        writer.startObject();
        writer.key("itype");
        writer.value(Collections.singletonList(alert.itype()));
        writer.key("ctype");
        writer.value(Collections.singletonList(alert.ctype()));
        if (!alert.nanny().isEmpty()) {
            writer.key("nanny");
            writer.value(alert.nanny());
        }
        if (alert.project() != null) {
            writer.key("prj");
            writer.value(Collections.singletonList(alert.project()));
        }
        writer.endObject();
        if (alert.window() != null) {
            writer.key("value_modify");
            writer.startObject();
            writer.key("window");
            writer.value(alert.window().duration());
            writer.key("type");
            writer.value(alert.window().aggregation());
            writer.endObject();
        }

        writer.key("warn");
        writer.startArray();
        writer.value(alert.warn().left());
        writer.value(alert.warn().right());
        writer.endArray();


        writer.key("crit");
        writer.startArray();
        writer.value(alert.crit().left());
        writer.value(alert.crit().right());
        writer.endArray();
        writer.key("juggler_check");
        writer.startObject();
        writer.key("host");
        writer.value(buildAlertHost(alert, config));
        writer.key("service");
        writer.value(buildAlertService(alert, config));
        if (config.namespace() != null) {
            writer.key("namespace");
            writer.value(config.namespace());
        }
        writer.key("tags");
        writer.value(alert.tags());
        writer.key("aggregator");
        writer.value("logic_or");
        writer.key("aggregator_kwargs");
        writer.startObject();
        writer.key("nodata_mode");
        writer.value("force_crit");
        writer.endObject();
        Integer stableTime = alert.stableTime();
        Integer critTime = alert.critTime();
        if (stableTime != null && critTime != null) {
            writer.key("flaps");
            writer.startObject();
            writer.key("boost");
            writer.value(0);
            writer.key("stable");
            writer.value(stableTime.intValue());
            writer.key("critical");
            writer.value(critTime.intValue());
            writer.endObject();
        }
        writer.key("pronounce");
        writer.value(alert.pronounce());
        writer.key("meta");
        writer.startObject();
        writer.key("urls");
        writer.startArray();
//        writer.value(JugglerUrl.alertUrl(alertName));
//        writer.value(JugglerUrl.screenshotUrl(alert.signal()));
//        writer.value(JugglerUrl.graphUrl(alert.signal()));
        if (alert.nanny() != null && !alert.nanny().isEmpty()) {
            int commonIndex = 0;
            int wordCommonIndex = -1;
            if (alert.nanny().size() > 1) {
                int minSize = alert.nanny().stream().map(String::length).min(Comparator.naturalOrder()).get();
                String first = alert.nanny().iterator().next();

                // dumb find common prefix
                boolean completed = false;
                while (!completed && commonIndex < minSize) {
                    char c = first.charAt(commonIndex);
                    if (!Character.isLetterOrDigit(c)) {
                        commonIndex += 1;
                        wordCommonIndex = commonIndex;
                        continue;
                    }
                    for (String nanny: alert.nanny()) {
                        if (c != nanny.charAt(commonIndex)) {
                            completed = true;
                            break;
                        }
                    }

                    if (!completed) {
                        commonIndex += 1;
                    }
                }

                for (String nanny: alert.nanny()) {
                    String name = nanny;
                    if (wordCommonIndex >= 0) {
                        name = nanny.substring(wordCommonIndex);
                    }
                    writer.value(
                        new JugglerUrl(
                            "https://nanny.yandex-team.ru/ui/#/services/catalog/" + nanny,
                            "nanny_" + name,
                            JugglerUrl.JugglerUrlType.NANNY));
                }
            } else {
                writer.value(
                    new JugglerUrl(
                        "https://nanny.yandex-team.ru/ui/#/services/catalog/" + config.nanny().iterator().next(),
                        "nanny",
                        JugglerUrl.JugglerUrlType.NANNY));
            }
        }

        if (alert.yasmPanel() != null) {
            writer.value(
                new JugglerUrl(
                    "https://yasm.yandex-team.ru/panel/" + alert.yasmPanel(),
                    "Панель",
                    JugglerUrl.JugglerUrlType.URL));
        }

        if (alert.yasmTemplate() != null) {
            StringBuilder sb =
                new StringBuilder(
                    "https://yasm.yandex-team.ru/template/panel/");
            sb.append(alert.yasmTemplate());
            if (alert.project() != null) {
                sb.append("/prj=");
                sb.append(alert.project());
            }
            if (!alert.nanny().isEmpty()) {
                sb.append(";nanny=");
                sb.append(String.join(",", alert.nanny()));
            }
            sb.append(";ctype=");
            sb.append(alert.ctype());
            sb.append('/');
            writer.value(
                new JugglerUrl(
                    new String(sb),
                    "Панель",
                    JugglerUrl.JugglerUrlType.URL));
        }
        if (alert.wiki() != null) {
            writer.value(
                new JugglerUrl(
                    alert.wiki(),
                    "wiki",
                    JugglerUrl.JugglerUrlType.WIKI));
        }
        if (alert.alertsConfigUri() != null) {
            writer.value(
                new JugglerUrl(
                    alert.alertsConfigUri(),
                    "alerts-config-uri",
                    JugglerUrl.JugglerUrlType.URL));
        }
        writer.endArray();
        writer.endObject();

        writer.key("notifications");
        writer.startArray();
        for (ImmutableNotificationConfig notification: alert.notifications()) {
            writeNotification(writer, notification);
        }

        for (String link: alert.notificationLinks()) {
            writeNotification(writer, config.notifyPresets().get(link));
        }
        writer.endArray();
        writer.endObject();
        writer.endObject();
    }

    private void writeNotification(
        final JsonWriter writer,
        final ImmutableNotificationConfig notification)
        throws Exception
    {
        writer.startObject();

        if (notification.method() == NotificationType.PHONE) {
            writer.key("template_name");
            writer.value("phone_escalation");
            writer.key("description");
            writer.nullValue();
            writer.key("template_kwargs");
            writer.startObject();
            writer.key("delay");
            writer.value(15);
            writer.key("logins");
            writer.value(notification.logins());
            if (notification.restartAfter() != null) {
                writer.key("restart_after");
                writer.value(notification.restartAfter());
            }

            if (notification.timeStart() != null) {
                writer.key("time_start");
                writer.value(
                    notification.timeStart().format(
                        DateTimeFormatter.ISO_TIME));
            }

            if (notification.timeEnd() != null) {
                writer.key("time_end");
                writer.value(
                    notification.timeEnd().format(DateTimeFormatter.ISO_TIME));
            }

            if (notification.dayStart() != null) {
                writer.key("day_start");
                writer.value(notification.dayStart());
            }

            if (notification.dayEnd() != null) {
                writer.key("day_end");
                writer.value(notification.dayEnd());
            }

            if (notification.repeat() != null) {
                writer.key("repeat");
                writer.value(notification.repeat());
            }

            if (notification.onSuccessNextCallDelay() != null) {
                writer.key("on_success_next_call_delay");
                writer.value(notification.onSuccessNextCallDelay());
            }
            writer.endObject();
        }

        if (notification.method() == NotificationType.TELEGRAM) {
            writer.key("template_name");
            writer.value("on_status_change");
            writer.key("description");
            writer.nullValue();
            writer.key("template_kwargs");
            writer.startObject();
            writer.key("login");
            writer.value(notification.logins());
            writer.key("method");
            writer.value(Collections.singletonList("telegram"));
            writer.key("status");
            writer.value(Arrays.asList("CRIT", "WARN", "OK"));

            if (notification.timeStart() != null) {
                writer.key("time_start");
                writer.value(
                    notification.timeStart().format(
                        DateTimeFormatter.ISO_TIME));
            }

            if (notification.timeEnd() != null) {
                writer.key("time_end");
                writer.value(
                    notification.timeEnd().format(DateTimeFormatter.ISO_TIME));
            }

            if (notification.dayStart() != null) {
                writer.key("day_start");
                writer.value(notification.dayStart());
            }

            if (notification.dayEnd() != null) {
                writer.key("day_end");
                writer.value(notification.dayEnd());
            }
            writer.endObject();
        }

        writer.endObject();
    }

    private void printCurrent(final ImmutableAlertsConfig config) throws Exception {
        HttpGet get = new HttpGet("/srvambry/alerts/list?name_prefix=" + buildAlertPrefix(config));
        try (CloseableHttpResponse response = yasmClient.execute(API_HOST, get)) {
            System.out.println(EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8));
        }
    }

    public static String buildAlertPrefix(final ImmutableAlertsConfig config) {
        if (config.module() != null) {
            return ALERTS_PREFIX + config.module();
        } else {
            return ALERTS_PREFIX;
        }

    }

    public static String buildAlertHost(
        final ImmutableAlertConfig alert,
        final ImmutableAlertsConfig config)
    {
        if (alert.host() != null) {
            return alert.host();
        }

        return config.module();
    }

    public static String buildAlertService(
        final ImmutableAlertConfig alert,
        final ImmutableAlertsConfig config)
    {
        if (alert.service() != null) {
            return alert.service();
        }

        return StringUtils.removePrefix(alert.name(), config.module() + '-');
    }

    public static void main(String[] args) throws Exception {

        Options options = new Options();

        Option input = new Option("c", "config", true, "alert conf file");
        input.setRequired(true);
        options.addOption(input);

        Option print = new Option("p", "print", false, "only print alerts");
        print.setRequired(false);
        options.addOption(print);

        Option list = new Option("l", "list", false, "only list current alerts");
        list.setRequired(false);
        options.addOption(list);

        CommandLineParser parser = new DefaultParser();
        HelpFormatter help = new HelpFormatter();
        CommandLine cmnd;
        try {
            cmnd = parser.parse(options, args);
        } catch (Exception e) {
            help.printHelp("alerts-uploader", options);
            return;
        }

        AlertsUploader uploader = new AlertsUploader();
        ImmutableAlertsConfig config =
            new AlertsConfigBuilder(
                new IniConfig(new File(cmnd.getOptionValue("config"))))
                .build();
        if (cmnd.hasOption("list")) {
            uploader.printCurrent(config);
            return;
        }

        uploader.upload(config, cmnd.hasOption("print"));
        uploader.printCurrent(config);
    }
}
