package ru.yandex.olelole.config;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;

public abstract class AbstractAlertConfigBuilder
    <T extends AbstractAlertConfigBuilder<T>>
     extends AbstractAlertMetaConfigBuilder<T> implements AlertConfig
{
    private String name;
    private String signal;
    private String pronounce;
    private DoubleInterval warn;
    private DoubleInterval crit;
    private Integer stableTime;
    private Integer critTime;
    private AlertWindowConfigBuilder window;
    private Set<String> notificationLinks;
    private List<NotificationConfigBuilder> notifications;;

    AbstractAlertConfigBuilder(final AlertConfig config) {
        super(config);
        name(config.name());
        signal(config.signal());
        warn(config.warn());
        crit(config.crit());
        stableTime(config.stableTime());
        critTime(config.critTime());
        window(config.window());
        notifications(config.notifications());
        notificationLinks(config.notificationLinks());
        pronounce(config.pronounce());
    }

    AbstractAlertConfigBuilder(
        final IniConfig config,
        final String name,
        final AlertConfig defaults,
        final AlertMetaConfig metaDefaults)
        throws ConfigException
    {
        super(config, metaDefaults);
        this.name = name;
        this.signal = config.getString("signal", defaults.signal());
        this.pronounce = config.getString("pronounce", defaults.pronounce());
        this.warn = config.get("warn", defaults.warn(), DoubleIntervalParser.INSTANCE);
        this.crit = config.get("crit", defaults.crit(), DoubleIntervalParser.INSTANCE);
        this.stableTime = config.get("stable-time", defaults.stableTime(), PositiveIntegerValidator.INSTANCE);
        this.critTime = config.get("crit-time", defaults.critTime(), PositiveIntegerValidator.INSTANCE);
        IniConfig windowSection = config.sectionOrNull("window");
        if (windowSection == null) {
            window = null;
        } else {
            this.window =
                new AlertWindowConfigBuilder(
                    windowSection,
                    defaults.window());
        }

        this.notifications = new ArrayList<>();
        IniConfig notificationSection = config.sectionOrNull("notify");
        if (notificationSection != null) {
            if (notificationSection.sections().isEmpty()) {
                this.notifications.add(new NotificationConfigBuilder(notificationSection));
            } else {
                for (Map.Entry<String, IniConfig> entry: notificationSection.sections().entrySet()) {
                    this.notifications.add(new NotificationConfigBuilder(entry.getValue()));
                }
            }
        }

        notificationLinks =
            config.get(
                "notifications",
                defaults.notificationLinks(),
                new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new));
    }

    @Override()
    public String name() {
        return name;
    }

    public T name(final String value) {
        this.name = value;
        return self();
    }

    @Override()
    public String signal() {
        return signal;
    }

    public T signal(final String value) {
        this.signal = value;
        return self();
    }


    @Override()
    public DoubleInterval warn() {
        return warn;
    }

    public T warn(final DoubleInterval value) {
        this.warn = value;
        return self();
    }

    @Override()
    public DoubleInterval crit() {
        return crit;
    }

    public T crit(final DoubleInterval value) {
        this.crit = value;
        return self();
    }

    @Override
    public Integer stableTime() {
        return stableTime;
    }

    public T stableTime(final Integer stableTime) {
        this.stableTime = stableTime;
        return self();
    }

    @Override
    public Integer critTime() {
        return critTime;
    }

    public T critTime(final Integer critTime) {
        this.critTime = critTime;
        return self();
    }

    @Override()
    public AlertWindowConfigBuilder window() {
        return window;
    }

    public T window(final AlertWindowConfig window) {
        if (window == null) {
            this.window = null;
        } else {
            this.window = new AlertWindowConfigBuilder(window);
        }
        return self();
    }

    @Override()
    public List<NotificationConfigBuilder> notifications() {
        return notifications;
    }

    public T notifications(final List<? extends NotificationConfig> value) {
        this.notifications = value.stream().map(NotificationConfigBuilder::new).collect(Collectors.toList());
        return self();
    }

    @Override
    public Set<String> notificationLinks() {
        return notificationLinks;
    }

    public T notificationLinks(final Set<String> notificationLinks) {
        this.notificationLinks = new LinkedHashSet<>(notificationLinks);
        return self();
    }
    @Override
    public String pronounce() {
        return pronounce;
    }

    public AbstractAlertConfigBuilder<T> pronounce(final String pronounce) {
        this.pronounce = pronounce;
        return this;
    }
}
