package ru.yandex.olelole.config;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractAlertsConfigBuilder
    <T extends AbstractAlertsConfigBuilder<T>>
    extends AbstractAlertMetaConfigBuilder<T>
    implements AlertsConfig
{
    private String module;
    private String namespace;
    private String abc;
    private Map<String, NotificationConfigBuilder> notifications;
    private Set<AlertConfigBuilder> alerts;

    AbstractAlertsConfigBuilder(final AlertsConfig config) {
        super(config);
        module(config.module());
        namespace(config.namespace());
        abc(config.abc());
        notifications(config.notifyPresets());
        alerts(config.alerts());
    }

    AbstractAlertsConfigBuilder(
        final IniConfig config,
        final AlertsConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        this.module = config.getString("module", defaults.module());
        this.namespace = config.getString("namespace", defaults.namespace());
        this.abc = config.getString("abc", defaults.abc());
        IniConfig notificationSection = config.sectionOrNull("notification");
        this.notifications = new LinkedHashMap<>();
        if (notificationSection != null) {
            for (Map.Entry<String, IniConfig> section: notificationSection.sections().entrySet()) {
                notifications.put(section.getKey(), new NotificationConfigBuilder(section.getValue()));
            }
        }

        IniConfig alertSection = config.sectionOrNull("alert");
        if (alertSection != null) {
            Set<AlertConfigBuilder> alerts = new LinkedHashSet<>();
            for (Map.Entry<String, IniConfig> entry: alertSection.sections().entrySet()) {
                alerts.add(new AlertConfigBuilder(entry.getValue(), entry.getKey(), this));
            }
            this.alerts = alerts;
        } else {
            this.alerts = Collections.emptySet();
        }
    }

    @Override()
    public String module() {
        return module;
    }

    public T module(final String value) {
        this.module = value;
        return self();
    }

    @Override()
    public String namespace() {
        return namespace;
    }

    public T namespace(final String value) {
        this.namespace = value;
        return self();
    }

    @Override()
    public String abc() {
        return abc;
    }

    public T abc(final String value) {
        this.abc = value;
        return self();
    }

    @Override()
    public Map<String, NotificationConfigBuilder> notifyPresets() {
        return notifications;
    }

    public T notifications(final Map<String, ? extends NotificationConfig>  value) {
        this.notifications = new LinkedHashMap<>();
        for (Map.Entry<String, ? extends NotificationConfig> entry: value.entrySet()) {
            this.notifications.put(entry.getKey(), new NotificationConfigBuilder(entry.getValue()));
        }
        return self();
    }

    @Override()
    public Set<AlertConfigBuilder> alerts() {
        return alerts;
    }

    public T alerts(final Set<? extends AlertConfig> value) {
        this.alerts = value.stream().map(AlertConfigBuilder::new).collect(Collectors.toSet());

        return self();
    }
}
