import argparse
import logging
from mail.tools.safely_delete_stids.lib import collect_stids, process_stids


DEFAULT_FILE_TO_LOG = '/var/log/safely_delete_stids.log'
DEFAULT_DBUSER = 'mxfront'
DEFAULT_TIMETAIL_TIMEOUT = 10


def main(log_file, time, timefmt, sharpei_host, dbuser, timeout, log_to_file):
    logging.basicConfig(filename=log_to_file, level=logging.DEBUG, format='%(asctime)s %(name)s %(message)s')
    logger = logging.getLogger('safely_delete_stids')
    logger.debug('Start working')
    try:
        stids_for_uid = collect_stids(log_file, time, timefmt, timeout)
    except RuntimeError as e:
        logger.exception('Collecting stids error: %s', e.message)

    logger.debug('Got %d uids/%d stids to process',
                 len(stids_for_uid),
                 sum(map(lambda l: len(l), stids_for_uid.itervalues()), 0))
    if stids_for_uid:
        count = process_stids(sharpei_host, dbuser, stids_for_uid)
        logger.debug('Successfully remove %d stids, exiting', count)
    else:
        logger.debug('No stids, my life is useless... Goodbye, cruel world!')


def parse_args():
    parser = argparse.ArgumentParser(description='Check log file and save unneeded stids into mail.deleted_queue')
    parser.add_argument('-l', '--logfile', dest='log_file', type=str, required=True,
                        help='Logfile for search unneeded stids in')
    parser.add_argument('--log-to', dest='log_to_file', type=str, required=False, default=DEFAULT_FILE_TO_LOG,
                        help='Logfile for logging')
    parser.add_argument('-t', '--time', dest='time', type=int, default=60,
                        help='Time interval(seconds) to perform search for')
    parser.add_argument('-f', '--timefmt', dest='timefmt', choices=('common', 'syslog', 'tskv'), required=True,
                        help='Log time format for timetail (use "syslog" for fastsrv and "common" for NSLS)')
    parser.add_argument('-s', '--sharpei-host', dest='sharpei_host', type=str, required=True,
                        help='Sharpei base url')
    parser.add_argument('-u', '--dbuser', dest='dbuser', type=str, default=DEFAULT_DBUSER, required=True,
                        help='DB user name')
    parser.add_argument('-T', '--timeout', dest='timeout', type=int, default=DEFAULT_TIMETAIL_TIMEOUT,
                        help='Timeout for getting log lines using timetail')
    return parser.parse_args()


if __name__ == '__main__':
    main(**dict(parse_args()._get_kwargs()))
