from collections import defaultdict
import logging
import os
import psycopg2
import re
import subprocess
from time import sleep
from mail.tools.safely_delete_stids.lib.sharpei import get_conninfo


SHARED_UID = '0'
CHUNK_SIZE = 10
# [2018-11-26 16:53:58.584709] (14826) {I} PNz8VU9hC7-0WpEfqeW-DLV should delete stid = 320.mail:1120000000098818.E1489433:789186892168144433819965785023
# Nov 26 00:00:05 mxback1o fastsrv: OvbDU4CuHuQ1: should delete stid = 320.mail:1120000000098818.E1489433:789186892168144433819965785023
STID_RE = re.compile('should delete stid = (?P<stid>\S+)')
logger = logging.getLogger('safely_delete_stids')


def collect_stids(log_file, time, timefmt, timeout):
    stids_for_uid = defaultdict(list)
    for line in get_maillog(log_file, time, timefmt, timeout):
        stid = get_stid(line)
        if stid is None:
            continue
        try:
            uid = get_uid_from_stid(stid)
        except Exception as e:
            logger.exception('Could not parse stid=%s: %s', stid, e.message)
            continue
        if uid == SHARED_UID:
            logger.debug('Skip shared stid=%s', stid)
            continue

        stids_for_uid[uid].append(stid)

    return stids_for_uid


def process_stids(sharpei_host, dbuser, stids_for_uid):
    count = 0
    total_users = len(stids_for_uid)
    cur_user_idx = 0
    for uid, stids in stids_for_uid.iteritems():
        cur_user_idx += 1
        logger.debug('[%d/%d] Process %d stids for uid=%s', cur_user_idx, total_users, len(stids), uid)
        try:
            conn_info = get_conninfo(sharpei_host, dbuser, uid)
        except Exception as e:
            logger.exception('Sharpei error for uid=%s: %s', uid, e.message)
            continue
        count_for_uid, failed_stids = remove_stids_via_db(conn_info, uid, stids)
        count += count_for_uid
        for stid in failed_stids:
            logger.error('Fail to delete stid=%s', stid)
        logger.info('Store in delete_queue %d stids for uid=%s', count_for_uid, uid)

    return count


def get_maillog(log_file='/var/log/maillog', time=60, timefmt='syslog', timeout=10):
    cmd = ['timetail', '-t', timefmt, '-n', str(time), log_file]
    with os.tmpfile() as log:
        process = subprocess.Popen(cmd, stdout=log, close_fds=True)
        duration = 0.0
        try:
            while process.poll() is None and duration < timeout:
                duration += 0.1
                sleep(0.1)
        finally:
            process.kill()
            process.wait()

        if duration >= timeout:
            raise RuntimeError('timetail timeout')

        log.seek(0)
        return log.readlines()


def get_stid(line):
    match = STID_RE.search(line)
    return match.group('stid') if match else None


def _make_pg_array(arr, elem_type='text'):
    return '\'{' + ','.join(arr) + '}\'::' + elem_type + '[]'


def remove_stids_via_db(conn_info, uid, stids):
    count = 0
    failed_stids = []
    try:
        connect = psycopg2.connect(conn_info, sslmode='require')
        cursor = connect.cursor()
    except Exception as e:
        logger.exception('Could not connect to db with connection string "%s" for uid=%s: %s',
                         conn_info, uid, e.message)
        return 0, stids

    try:
        for i in range(0, len(stids), CHUNK_SIZE):
            cur_stids = stids[i:i + CHUNK_SIZE]
            str_stids = _make_pg_array(cur_stids)
            try:
                query = '''
                    SELECT stid
                    FROM UNNEST({stids}) AS stid
                    WHERE EXISTS (
                        SELECT 1
                        FROM mail.storage_delete_queue AS q
                        WHERE q.uid={uid} AND q.st_id=stid)
                '''.format(uid=uid, stids=str_stids)
                cursor.execute(query)
                for stid, in cursor:
                    cur_stids.remove(stid)
                    count += 1  # treat it as successfully deleted

                if not cur_stids:
                    continue

                str_stids = _make_pg_array(cur_stids)
                query = 'SELECT code.add_to_storage_delete_queue({uid}::bigint, {stids})'.format(uid=uid, stids=str_stids)
                cursor.execute(query)
                connect.commit()
            except Exception as e:
                logger.exception('DB error: %s', e.message)
                failed_stids += cur_stids
                continue

            count += len(cur_stids)
    finally:
        connect.close()
    return count, failed_stids


def get_uid_from_stid(stid):
    parts = stid.split('.')
    if len(parts) != 3:
        raise ValueError('Malformed stid')
    if not parts[1].startswith('mail:'):
        raise ValueError('Possibly not mail stid')
    return parts[1].rsplit(':', 1)[1]
