import argparse
import sys

from tqdm import tqdm

from mail.tools.sql_execute_per_shard.lib.get_shards import get_sharpei_response, get_master_dns, get_password_from_yav
from mail.tools.sql_execute_per_shard.lib.pg_helper import execute_query, execute_select


def main():
    parser = argparse.ArgumentParser(prog="Run sharpei on all shards",
                                     description="""
    Program can be run both as interactive or non-interactive.
    In case of non-interactive mode, it is recommended to redirect
    standard output to the file, so that it does not mess with the progress
    bar.""")
    parser.add_argument("--sharpei_host", type=str, required=True,
                        help="Sharpei host")
    parser.add_argument("--sharpei_access_machine", type=str, required=True,
                        help="Machine that has access to sharpei via net")
    parser.add_argument("--db_user", type=str, required=True,
                        help="Database user")
    parser.add_argument("--sql_file_name", type=str, required=True,
                        help="The sql file to execute on each shard")
    parser.add_argument("--interactive", type=str, choices=["yes", "no"],
                        default="yes",
                        help="Should program wait for user input after each node?")
    parser.add_argument("--query_select", type=str, choices=["yes", "no"],
                        default="yes", help="Is this simply a select query?")

    args = parser.parse_args()

    sharpei_host = args.sharpei_host
    sharpei_access_machine = args.sharpei_access_machine
    db_user = args.db_user
    sql_file_name = args.sql_file_name

    with open(sql_file_name, "r") as file:
        query = file.read()

    print(f"""The query to be executed:
    ====
    {query}
    ====""")

    response = get_sharpei_response(sharpei_host, sharpei_access_machine)
    db_password = get_password_from_yav(db_user, sharpei_host)
    dns = get_master_dns(response, db_user, db_password)

    # Если процесс интерактивный, то выводить progressbar бессмыссленно
    if args.interactive == "no":
        dns = tqdm(iterable=dns)

    for shard, url in dns:
        print(f"Processing {shard}...")
        if args.interactive == "yes":
            print("Press ENTER to continue...")
            input()
        if args.query_select == "yes":
            execute_select(url, query)
        else:
            execute_query(url, query)
        sys.stdout.flush()

if __name__ == '__main__':
    main()
