import logging
from datetime import datetime, timedelta
from typing import Sequence
import os.path

from startrek_client import Startrek
from startrek_client.objects import Resource

from lead_helpers.tracker.issues import *
from lead_helpers.tracker.network_diagram import *
from lead_helpers.tracker.schedule import Scheduler
from lead_helpers.tracker.schedule.choose_policy import WeightPolicy
from lead_helpers.tracker.schedule.duration_policy import SkipHolidaysMixin, StoryPointSchedulePolicy
from lead_helpers.tracker.schedule.scheduler import ScheduledIssue

logging.basicConfig(level=logging.WARNING)
logging.getLogger('lead_helpers').setLevel(logging.DEBUG)


STORYPOINTS_TO_DURATION = {
    None: timedelta(days=7),
    0.0: timedelta(days=14),
    0.5: timedelta(days=1) * 0.5,
    1: timedelta(days=1),
    2: timedelta(days=2),
    3: timedelta(days=4),
    5: timedelta(days=7),
    8: timedelta(days=14),
}


def local_client(token_path=os.path.expanduser('~/.oauth/startrek')):
    return Startrek(
        useragent='prez @ Jupyter Notebook',
        token=open(token_path).read().strip(),
    )


class DurationPolicy(SkipHolidaysMixin, StoryPointSchedulePolicy):
    pass


def schedule_maildev_team(client, team_queues: Sequence[str], epic_keys: Sequence[str], start_time: datetime, max_parallel=3):
    queues = [client.queues[key] for key in team_queues]

    # Skip issues in team-foreign queues
    def managed_by_team(issue):
        return issue.queue.key in [q.key for q in queues]

    epics: Sequence[Resource] = [client.issues[key] for key in epic_keys]
    issues = expand(client, subepics(client, epics), issue_filter=managed_by_team)
    edges = dep_graph(client, issues)

    scheduler = Scheduler(
        client=client,
        # Calc issue duration from a story points
        duration_policy=DurationPolicy(sp_to_duration=STORYPOINTS_TO_DURATION),
        # Prioritize issues that are on critical path
        choose_policy=WeightPolicy.by_critical_path(
            issues=issues,
            deps=edges,
            init_weights={i.key: STORYPOINTS_TO_DURATION[i.storyPoints] for i in issues}
        ),
    )
    return list(scheduler.plan(issues, edges, start_time, max_parallel))


def update_schedule(issue_times: Sequence[ScheduledIssue], dry_run: bool = False):
    for scheduled_issue in issue_times:
        print(f'[{scheduled_issue.issue.key}] is scheduled at {scheduled_issue.start} - {scheduled_issue.end}')
        if not dry_run:
            scheduled_issue.issue.update(
                start=scheduled_issue.start.date().isoformat(),
                end=(scheduled_issue.end - timedelta(seconds=1)).date().isoformat()
            )
