import logging
from itertools import chain, groupby

from .issues import depends_on

log = logging.getLogger(__name__)


def dep_graph(client, issues):
    """Build dependency graph between set of issues based on `depends_on` relation"""
    issue_keys = {i.key for i in issues}
    return {
        issue.key: {
            link.object.key for link in issue.links
            if link.object.key in issue_keys and depends_on(client, link)
        }
        for issue in issues
    }


def expand(client, issues, issue_filter=None):
    """
    Enrich set of issues with any blocking issues.
    """
    # Naive implementation that scans through all dependencies of current set of issues until no new issues are found.
    while True:
        ret = list(filter(issue_filter, _expand_step(client, issues)))
        log.debug([issue.key for issue in ret])
        if len(ret) == len(issues):
            break
        issues = ret
    return issues


def _expand_step(client, issues):
    def key(issue):
        return issue.key

    return [
        # Ugly replacement for sort | uniq
        next(group) for _, group in groupby(
            sorted(
                issues + [
                    # Take all issues that are dependent from any of current issues
                    link.object for link
                    in chain.from_iterable(issue.links for issue in issues)
                    if depends_on(client, link)
                ],
                key=key
            ),
            key=key
        )
    ]
