from abc import ABCMeta, abstractmethod
from typing import Mapping, Iterable, Any, Collection

from toposort import toposort_flatten

from lead_helpers.tracker.typing import Issue, IssueKey


class ChoosePolicy(metaclass=ABCMeta):
    @abstractmethod
    def next_issue(self, issues: Iterable[Issue]) -> Issue:
        """Take next issue, rules are defined by policy implementation"""
        pass


class WeightPolicy(ChoosePolicy):
    def __init__(self, weights: Mapping[IssueKey, Any]):
        self._weights = weights

    @classmethod
    def by_critical_path(
            cls,
            issues: Iterable[Issue],
            deps: Mapping[IssueKey, Collection[IssueKey]],
            init_weights: Mapping[IssueKey, Any],
    ):
        """For any issue I, weight(I) = init_weight(I) + sum(weight(J) for all J such that I is in deps(J))"""
        assert all(issue.key in deps for issue in issues)
        assert all(issue.key in init_weights for issue in issues)

        weights = dict(**init_weights)
        end_to_start = (x for x in toposort_flatten(deps))
        """Use topological sorting to propagate weights from end of dependency graph to start"""
        for issue in end_to_start:
            for dep in deps[issue]:
                weights[dep] += weights[issue]
        return cls(weights=weights)

    def next_issue(self, issues: Iterable[Issue]) -> Issue:
        try:
            return max(issues, key=lambda i: self._weights[i.key])
        except KeyError as e:
            key = e.args[0]
            raise AssertionError(f'Issue {key} not found in weights mapping {self._weights}')
