from abc import abstractmethod, ABCMeta
from datetime import datetime, timedelta
from typing import Dict

from lead_helpers.tracker.typing import Issue


class DurationPolicy(metaclass=ABCMeta):
    @abstractmethod
    def calc_work_end(self, issue: Issue, start_ts: datetime) -> datetime:
        pass


class StoryPointSchedulePolicy(DurationPolicy):
    def __init__(self, sp_to_duration: Dict[int, timedelta]):
        self.sp_to_duration = sp_to_duration

    def calc_work_end(self, issue: Issue, start_ts: datetime) -> datetime:
        return start_ts + self.sp_to_duration[issue.storyPoints]


class SkipHolidaysMixin(DurationPolicy):
    def calc_work_end(self, issue: Issue, start_ts: datetime) -> datetime:
        end_ts = super().calc_work_end(issue=issue, start_ts=start_ts)
        while end_ts.isoweekday() in (6, 7):
            end_ts += timedelta(days=1)
        return end_ts
