import logging
from dataclasses import dataclass
from datetime import timedelta, datetime
from typing import Iterable

from yandex_tracker_client import TrackerClient

from .choose_policy import ChoosePolicy
from .duration_policy import DurationPolicy
from .state import State
from .workers import Workers
from lead_helpers.tracker.issues import depends_on, is_blocker_for
from ..typing import Issue

log = logging.getLogger(__name__)


@dataclass
class ScheduledIssue:
    issue: Issue
    start: datetime
    end: datetime


@dataclass
class Scheduler:
    client: TrackerClient
    duration_policy: DurationPolicy
    choose_policy: ChoosePolicy

    def plan(self, issues, edges, start_time, max_parallel) -> Iterable[ScheduledIssue]:
        ts = start_time
        issues_by_key = {i.key: i for i in issues}
        state = State(pending={key: issues_by_key[key] for key, deps in edges.items() if not deps})
        workers = Workers(max_parallel, start_time)

        while len(state.done) < len(issues):
            state.check_issues()
            assert set(state.in_proc.keys()) == set(worker.issue.key for worker in workers.busy_workers)

            if state.pending and workers.free_workers:
                issue, end_ts = self._schedule_issue(state, workers, ts)
                yield ScheduledIssue(issue, ts, end_ts)
            elif workers.busy_workers:
                ts = workers.nearest_end()
                for end_ts, issue in workers.process_to(ts):
                    self._end_scheduled_issue(edges, state, issue)
            else:
                raise AssertionError('No ready issues, no in_progress issues, not all issues are done')

    def _schedule_issue(self, state: State, workers: Workers, ts):
        """Take most important issue to plan"""
        issue = self.choose_policy.next_issue(state.pending.values())
        i_end = self.duration_policy.calc_work_end(issue, ts)

        state.start(issue.key)
        workers.work_till(i_end, issue)

        return issue, i_end

    def _end_scheduled_issue(self, edges, state: State, issue):
        """Move to nearest end of in_proc issues"""
        state.end(issue.key)
        # Add ready deps of done issue
        # TODO :: Check by edges, not by links
        for link in issue.links:
            if link.object.key in edges and is_blocker_for(self.client, link) and self.all_deps_done(link.object, state):
                state.add(link.object)

    def all_deps_done(self, issue, state: State):
        return all(
            ilink.object.key in state.done
            for ilink in issue.links
            if depends_on(self.client, ilink)
        )
