import logging
from dataclasses import dataclass, field
from typing import Dict

from lead_helpers.tracker.typing import Issue, IssueKey

log = logging.getLogger(__name__)


@dataclass
class State:
    pending: Dict[IssueKey, Issue]
    in_proc: Dict[IssueKey, Issue] = field(default_factory=dict, init=False)
    done: Dict[IssueKey, Issue] = field(default_factory=dict, init=False)

    def add(self, issue: Issue):
        assert issue.key not in self.pending
        self.pending[issue.key] = issue
        log.info(f'{issue.key}: NONE -> Pending')

    def start(self, ikey: IssueKey):
        issue = self.pending.pop(ikey, None)
        if issue is None:
            raise AssertionError(f'Issue {ikey} not found in pending issues {self.pending}')
        assert issue.key not in self.in_proc
        self.in_proc[issue.key] = issue
        log.info(f'{ikey}: Pending -> InProcess')

    def end(self, ikey: IssueKey):
        issue = self.in_proc.pop(ikey, None)
        if issue is None:
            raise AssertionError(f'Issue {ikey} not found in in-process issues {self.in_proc}')
        assert issue.key not in self.done
        self.done[issue.key] = issue
        log.info(f'{ikey}: InProcess -> Done')

    def check_issues(self):
        pending_keys = set(self.pending.keys())
        in_proc_keys = set(self.in_proc.keys())
        done_keys = set(self.done.keys())
        log.debug(f'Pending: {pending_keys}')
        log.debug(f'In-process: {in_proc_keys}')
        log.debug(f'Done: {done_keys}')
        assert pending_keys & in_proc_keys == set()
        assert pending_keys & done_keys == set()
        assert in_proc_keys & done_keys == set()

