from dataclasses import dataclass
from datetime import datetime
from typing import Optional, Iterable, Tuple

from yandex_tracker_client.objects import Resource as Issue


@dataclass
class Worker:
    end_at: datetime
    issue: Optional[Issue] = None

    @property
    def busy(self):
        return self.issue is not None


class Workers:
    def __init__(self, n: int, start_ts: datetime):
        self._workers = [Worker(end_at=start_ts) for _ in range(n)]

    @property
    def free_workers(self):
        return [w for w in self._workers if not w.busy]

    @property
    def busy_workers(self):
        return sorted([w for w in self._workers if w.busy], key=lambda w: w.end_at)

    def done_workers(self, ts: datetime):
        return [w for w in self.busy_workers if w.end_at <= ts]

    def process_to(self, ts: datetime) -> Iterable[Tuple[datetime, Issue]]:
        for worker in self.done_workers(ts):
            issue = worker.issue
            worker.issue = None
            yield worker.end_at, issue

    def work_till(self, end_ts: datetime, issue: Issue):
        assert self.free_workers, 'No free workers available'
        worker = self.free_workers[0]
        worker.end_at = end_ts
        worker.issue = issue

    def nearest_end(self) -> datetime:
        return next((w.end_at for w in self.busy_workers), None)
