# coding: utf-8

import logging
import os
from argparse import ArgumentParser

from mail.tools.yamail_info.lib.blackbox import (
    make_bb_url,
    parse_bb_resonse,
    find_uid_in_bb,
    NotAMailUser,
)
from mail.tools.yamail_info.lib.config import (
    get_config_location,
    load_config,
    write_default_config,
)
from mail.tools.yamail_info.lib.environment import make_env
from mail.tools.yamail_info.lib.utils import init_logging, common_arguments

log = logging.getLogger(__name__)


def bb_url_by_args(env, args):
    for arg_name, sid in [
            ('login', 'smtp'),
            ('uid', '2'),
    ]:
        arg_value = getattr(args, arg_name)
        if arg_value is not None:
            return make_bb_url(env, arg_name, arg_value, sid)


def main():
    parser = ArgumentParser(parents=[common_arguments()])
    parser.add_argument(
        '-l', '--logins-file',
        metavar='LOGINS',
        help='Path to input file with logins, one per line',
        required=True,
    )
    parser.add_argument(
        '-u', '--uids-file',
        metavar='UIDS',
        help='Path to output file with uids, one per line',
        required=True,
    )

    args = parser.parse_args()
    init_logging(args.verbosity)

    if not os.path.exists(get_config_location()):
        write_default_config(get_config_location())
        return 1
    env = make_env(args.env, args.db_user, args.arcadia_root, load_config(get_config_location()))

    with open(args.logins_file, 'r') as ifd:
        with open(args.uids_file, 'w') as ofd:
            def write_line(login, uid):
                ofd.write('{},{}\n'.format(login, uid))

            write_line('login', 'uid')
            for login in map(lambda s: s.strip(), ifd.readlines()):
                url = make_bb_url(env, by_field='login', by_value=login, sid='smtp')
                ticket = env.get_blackbox_service_ticket()
                bb_tree = parse_bb_resonse(env.get_url(url, service_ticket=ticket))
                try:
                    uid = find_uid_in_bb(bb_tree)
                except NotAMailUser as exc:
                    log.warning(exc)
                    uid = ''
                write_line(login, uid)


if __name__ == '__main__':
    main()
