from dataclasses import dataclass, asdict
from enum import Enum


class EnvironmentTypes(Enum):
    PROD = 'prod'
    TEST = 'test'
    CORP = 'corp'
    LOAD = 'load'
    QA = 'qa'
    CORP_QA = 'corp-qa'

    def __repr__(self):
        return f'{self.value}'


@dataclass
class Environment:
    yav_filename: str = None
    cluster_id: int = None
    db_name: str = None


@dataclass
class DbParams:
    secret_id: str = None
    default_user: str = None
    envs: dict = None


HUSKYDB = DbParams(secret_id='sec-01dm8dc1qsb8csvmbhfg9ksnv9', default_user='husky', envs={
    EnvironmentTypes.PROD: Environment('prod.yaml', '58afb252-b83e-43b3-9949-d12c4b0c9d4d', 'huskydb'),
    EnvironmentTypes.CORP: Environment('corp.yaml', 'f31d3eb0-040b-4fc8-9dc1-338eb9db8e51', 'huskydb_intranet'),
    EnvironmentTypes.LOAD: Environment('load.yaml', 'mdb7mrj7hkstveauf28g', 'huskydb'),
    EnvironmentTypes.TEST: Environment('test.yaml', 'f377c83d-618d-4c42-b3b3-6713df472be9', 'huskydb_test'),
})

SPANIELDB = DbParams(secret_id='sec-01fdz8scmkvpet6zejzfvm7crp', default_user='spaniel', envs={
    EnvironmentTypes.PROD: Environment('prod.yaml', 'mdbqpl3a2tftham7ikuh', 'spanieldb'),
    EnvironmentTypes.TEST: Environment('test.yaml', 'mdbk87lgmrido9bdcge0', 'spanieldb'),
})

QUEUEDB = DbParams(secret_id='sec-01ewdkqrmahc3e34e35sd9add4', default_user='barbet', envs={
    EnvironmentTypes.QA: Environment('prestable.yaml', 'mdb0ue49q2k5le0rvsil', 'queuedb'),
    EnvironmentTypes.PROD: Environment('prod.yaml', 'mdbvd6rdmijd2tkjr509', 'queuedb'),
    EnvironmentTypes.TEST: Environment('test.yaml', 'mdb1asfv04n0eo6pm1m2', 'queuedb'),
})

CALLMEBACKDB = DbParams(secret_id='sec-01eayp266v37ky4kymq07s0fzv', default_user='callmeback', envs={
    EnvironmentTypes.PROD: Environment('prod.yaml', '6a12b272-08ce-417d-a208-5ceac6293ba6', 'callmebackdb_prod'),
    EnvironmentTypes.TEST: Environment('test.yaml', '482d6b0a-a096-4692-b5c8-4af1680064d2', 'callmebackdb_test'),
})

SHARDDB = DbParams(secret_id='sec-01g8dyrt2zmrcgp7wwr3n34hk4', default_user='sharpei', envs={
    EnvironmentTypes.PROD: Environment('prod.yaml', 'mdbd8ebfpfh8ult1t721', 'sharddb'),
    EnvironmentTypes.CORP: Environment('corp.yaml', 'mdbm78cpbp77blunrfpo', 'sharddb'),
    EnvironmentTypes.LOAD: Environment('load.yaml', 'mdb89eo15641i2ooopcf', 'sharddb'),
    EnvironmentTypes.TEST: Environment('test.yaml', 'mdbfi97nl13865k7lfs7', 'sharddb'),
})

COLLIEDB = DbParams(secret_id='sec-01g8dwm4dyvfczf7eeggwr9k06', default_user='collie', envs={
    EnvironmentTypes.PROD: Environment('prod.yaml', 'mdbso7mulldk6283bhhe', 'collie_db'),
    EnvironmentTypes.TEST: Environment('test.yaml', 'mdbvn4cnlud4tdi3tnrq', 'collie_db_test'),
})

MOPSDB = DbParams(secret_id='sec-01g8dzwz4ec03qac3w86jb07k8', default_user='mops', envs={
    EnvironmentTypes.PROD: Environment('prod.yaml', 'mdbm8poisf3aefa4bk1v', 'mopsdb'),
    EnvironmentTypes.CORP: Environment('corp.yaml', 'mdbiu006uv3uvoi2791i', 'mopsdb'),
    EnvironmentTypes.LOAD: Environment('load.yaml', 'mdb9gu9a659vajjvoc32', 'mopsdb'),
    EnvironmentTypes.TEST: Environment('test.yaml', 'mdbr7r784pshojfeb0uh', 'mopsdb'),
    EnvironmentTypes.QA: Environment('qa.yaml', 'mdbjr3fs0g3tqa2ald7s', 'mopsdb'),
    EnvironmentTypes.CORP_QA: Environment('corp-qa.yaml', 'mdb1il09tq2hlg1sfffl', 'mopsdb'),
})

CACHEDB = DbParams(secret_id='sec-01g8dxg7j1an17h87rcxdh9dan', default_user='cachedb', envs={
    EnvironmentTypes.PROD: Environment('prod.yaml', 'mdbro0nkrsr3nnk5srsl', 'cachedb'),
    EnvironmentTypes.TEST: Environment('test.yaml', 'mdb7vntphipnbi8mi6ls', 'cachedb'),
})

DATABASES = {
    'huskydb': HUSKYDB,
    'spanieldb': SPANIELDB,
    'queuedb': QUEUEDB,
    'callmebackdb': CALLMEBACKDB,
    'sharddb': SHARDDB,
    'colliedb': COLLIEDB,
    'mopsdb': MOPSDB,
    'cachedb': CACHEDB,
}


def get_configs_description(indent):
    def get_configs_description_recursively(data, level):
        descr = ''

        if not isinstance(data, dict):
            return repr(data)

        for k, v in data.items():
            descr += f"\n{level * indent * ' '}{repr(k)}: {get_configs_description_recursively(v, level+1)}"

            if level == 0:
                descr += '\n'

        return descr

    data = dict()

    for db_name, db_config in DATABASES.items():
        data[db_name] = asdict(db_config)
        data[db_name]['secret_id'] = '***'

    return get_configs_description_recursively(data=data, level=0)
