import logging
import subprocess
import yaml
import urllib.parse

from mail.tools.yamail_info.bin.ya_db.configs import DATABASES
from library.python.vault_client.instances import Production as VaultClient
from library.python.vault_client.errors import ClientError

log = logging.getLogger(__name__)


class Database:
    def __init__(self, env, db, user=None):
        self.config = DATABASES[db]
        self.env = self.config.envs[env]
        self.user = user if user else self.config.default_user

    def connect(self, mode):
        password = self.__get_password_from_yav(self.config.secret_id, self.user, self.env)
        url = self.__get_url(self.env.cluster_id, mode)
        psql = self.__get_connection_command(url, self.user, self.env.db_name, password)

        log.info('Execute %r', self.__get_connection_command(url, self.user, self.env.db_name, '***'))
        popen = subprocess.Popen(psql, shell=True)
        while True:
            try:
                popen.communicate()
            except KeyboardInterrupt:
                # Mimic psql-behaviour to flush the prompt on CTRL+C rather than quit
                continue
            break

    @staticmethod
    def __get_connection_command(url, user, db, password):
        encoded_password = Database.__transform_representation_of_nix_shell_special_chars(password)

        return f'psql postgresql://{user}:{encoded_password}@{url}/{db}?port=6432'

    @staticmethod
    def __get_url(cluster_id, mode):
        return f'c-{cluster_id}.{mode}.db.yandex.net'

    @staticmethod
    def __get_password_from_yav(secret_id, user, env):
        client = VaultClient(decode_files=True)

        try:
            secret = client.get_version(secret_id)
        except ClientError as e:
            log.error('Failed to get secret from yav: %s', e)
            raise

        if env.yav_filename in secret['value']:
            secret_yaml = yaml.safe_load(secret['value'][env.yav_filename])
            try:
                password = secret_yaml[user]['password']
            except KeyError:
                raise Exception('Password for user {} not found in secret'.format(user))
        else:
            raise Exception('Environment {} not found in secret'.format(env.yav_filename))

        return password

    @staticmethod
    def __transform_representation_of_nix_shell_special_chars(data):
        return urllib.parse.quote(data)
