# coding: utf-8

import os
import sys

from argparse import ArgumentParser

from mail.tools.yamail_info.lib.config import (
    get_config_location,
    load_config,
    write_default_config,
)
from mail.tools.yamail_info.lib.environment import make_env
from mail.tools.yamail_info.lib.utils import (
    init_logging,
    pretty_json,
    get_query,
    common_arguments,
)
from mail.tools.yamail_info.lib.blackbox import (
    make_bb_url,
    parse_bb_resonse,
    find_uid_in_bb,
    NotAMailUser,
)
from mail.tools.yamail_info.lib import sharpei


def bb_url_by_args(env, args):
    for arg_name, sid in [
            ('login', 'smtp'),
            ('uid', '2'),
    ]:
        arg_value = getattr(args, arg_name)
        if arg_value is not None:
            return make_bb_url(env, arg_name, arg_value, sid)


def main():
    parser = ArgumentParser(parents=[common_arguments()])
    user_options = parser.add_mutually_exclusive_group(
        required=True
    )
    user_options.add_argument(
        '-l', '--login',
        metavar='LOGIN',
        help='help mail login',
    )
    user_options.add_argument(
        '-u', '--uid',
        metavar='UID',
        type=int,
        help='yandex uid',
    )
    sql_options = parser.add_mutually_exclusive_group(
        required=False
    )
    sql_options.add_argument(
        '-r',
        '--run-shell',
        action='store_true',
        help='start psql shell'
    )
    sql_options.add_argument(
        '--sql-query',
        help='exeute sql command in psql'
    )

    parser.add_argument(
        '--master',
        dest='use_master',
        action='store_true',
        help='run psql on master'
    )

    args = parser.parse_args()

    init_logging(args.verbosity)

    if not os.path.exists(get_config_location()):
        write_default_config(get_config_location())
        return 1
    env = make_env(args.env, args.db_user, args.arcadia_root, load_config(get_config_location()))

    ticket = env.get_blackbox_service_ticket()
    bb_resp = env.get_url(bb_url_by_args(env, args), service_ticket=ticket)
    print(pretty_json(bb_resp))

    bb_tree = parse_bb_resonse(bb_resp)

    try:
        uid = find_uid_in_bb(bb_tree)
    except NotAMailUser as exc:
        print(str(exc))
        return

    shapei_data = env.get_url(sharpei.conninfo_url(env.SHARPEI, uid))
    print(pretty_json(shapei_data))
    xdb = sharpei.parse_conninfo(shapei_data)
    if args.run_shell or args.sql_query:
        host = xdb.master if args.use_master else xdb.replica
        env.run_psql(host.address, uid, get_query(args.sql_query))


if __name__ == '__main__':
    sys.exit(main())
