#!/usr/local/bin/python

import os
import subprocess
import logging
import yaml

from ticket_parser2.api.v1 import BlackboxClientId
from library.python.vault_client.instances import Production as VaultClient
from library.python.vault_client.errors import ClientError

from mail.shiva.client.shiva import ShivaClient


log = logging.getLogger(__name__)


def make_pg_dsn(xdb_dsn, user):
    dsn_vars = dict(kv.split('=') for kv in xdb_dsn.split())
    return 'postgresql://{user}@{host}:{port}/{dbname}'.format(
        user=user,
        **dsn_vars
    )


def make_psql_command(dsn, uid, sql):
    cmdl = ['psql']
    if uid:
        cmdl.append('--variable=uid=%d' % uid)
    cmdl.append(dsn)
    if sql:
        cmdl = ['echo "' + sql + '"|'] + cmdl
    log.debug('sql is %r psql command is %r', sql, cmdl)
    return ' '.join(cmdl)


def get_password_from_yav(psql_user, env):
    secret_id = 'sec-01d5v5dv67pza174zneyt5s1tf'

    name = {
        'prod': 'users.yaml',
        'testing': 'users_test.yaml',
        'load': 'users_load.yaml',
        'corp': 'users_corp.yaml'
    }[env]

    client = VaultClient(decode_files=True)

    try:
        secret = client.get_version(secret_id)
    except ClientError as e:
        log.error('Failed to get secret from yav: %s', e)
        raise

    if name in secret['value']:
        secret_yaml = yaml.safe_load(secret['value'][name])
        try:
            password = secret_yaml[psql_user]['password']
        except KeyError:
            raise Exception('Password for user {} not found in secret'.format(psql_user))
    else:
        raise Exception('Environment {} not found in secret'.format(name))

    return password


def _get_service_ticket(arcadia_root, src, dst):
    args = [
        os.path.join(arcadia_root, 'ya'),
        'tool', 'tvmknife',
        'get_service_ticket', 'sshkey',
        '--src', src, '--dst', dst
    ]

    log.info('Local execute %r', args)
    cmd = subprocess.Popen(
        args,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
    )
    (out, err) = cmd.communicate()
    if cmd.returncode != 0:
        log.warning(
            'Command %r exit with %r: error: %s',
            args, cmd.returncode, err)
    return out.strip().decode('UTF-8')


def _get_shard_id_from_shiva(cluster_id, shiva_loc):

    shiva = ShivaClient(
        location=shiva_loc,
        logger=log
    )
    shard_json = shiva.admin().shards(cluster_id=cluster_id).json()
    shard_id = list(shard_json.keys())[0]

    return shard_id


class Prod(object):
    BLACKBOX = 'http://blackbox.yandex.net/blackbox'
    SHARPEI = 'http://sharpei-production.mail.yandex.net'
    SELF_TVM_ID = '2019581'
    BLACKBOX_TVM_ID = BlackboxClientId.Prod.value
    SHIVA_LOC = 'http://shiva-prod.mail.yandex.net'

    def __init__(self, env_type, web_host, psql_user, arcadia_root, psql_host=None):
        self.env = env_type
        self.psql_host = psql_host
        self.psql_user = psql_user
        self.psql_password = get_password_from_yav(self.psql_user, self.env)
        self.web_host = web_host
        self.arcadia_root = arcadia_root

    def get_blackbox_service_ticket(self):
        return _get_service_ticket(self.arcadia_root, self.SELF_TVM_ID, self.BLACKBOX_TVM_ID)

    def get_shard_id_from_shiva(self, cluster_id):
        return _get_shard_id_from_shiva(cluster_id, self.SHIVA_LOC)

    def get_url(self, url, service_ticket=None):
        curl = 'curl -v "%s"' % url
        if service_ticket:
            curl += ' -H "X-Ya-Service-Ticket: %s"' % service_ticket

        args = [
            'ssh', self.web_host, curl
        ]

        log.info('Execute on web %r', args)
        cmd = subprocess.Popen(
            args,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
        (out, err) = cmd.communicate()
        if cmd.returncode != 0:
            log.warning(
                'Command %r exit with %r: error: %s',
                args, cmd.returncode, err)
        return out

    def run_psql(self, xdb_address, uid=None, sql=None):
        if self.psql_host:
            args = [
                '/usr/bin/ssh',
                '-tt',
                self.psql_host,
                make_psql_command(xdb_address.dsn(self.psql_user, self.psql_password), uid, sql),
            ]
            log.info('Execute %r on remote host %r', make_psql_command(xdb_address.dsn(self.psql_user, '***'), uid, sql), self.psql_host)
            subprocess.Popen(args).communicate()
        else:
            psql = make_psql_command(xdb_address.dsn(self.psql_user, self.psql_password), uid, sql)
            log.info('Execute %r', make_psql_command(xdb_address.dsn(self.psql_user, '***'), uid, sql))
            popen = subprocess.Popen(psql, shell=True)
            while True:
                try:
                    popen.communicate()
                except KeyboardInterrupt:
                    # Mimic psql-behaviour to flush the prompt on CTRL+C rather than quit
                    continue
                break


class Corp(Prod):
    BLACKBOX = 'http://blackbox.yandex-team.ru/blackbox'
    SHARPEI = 'http://sharpei-intranet-production.mail.yandex.net'
    SELF_TVM_ID = '2019583'
    BLACKBOX_TVM_ID = BlackboxClientId.ProdYateam.value
    SHIVA_LOC = 'http://shiva-corp.mail.yandex.net'


class Testing(object):
    BLACKBOX = 'http://pass-test.yandex.ru/blackbox'
    SHARPEI = 'http://sharpei-testing.mail.yandex.net'
    SELF_TVM_ID = '2019585'
    BLACKBOX_TVM_ID = BlackboxClientId.Test.value
    SHIVA_LOC = 'http://shiva-test.mail.yandex.net'

    def __init__(self, env_type, psql_user, arcadia_root):
        self.env = env_type
        self.psql_user = psql_user
        self.psql_password = get_password_from_yav(self.psql_user, self.env)
        self.arcadia_root = arcadia_root

    def get_blackbox_service_ticket(self):
        return _get_service_ticket(self.arcadia_root, self.SELF_TVM_ID, self.BLACKBOX_TVM_ID)

    def get_shard_id_from_shiva(self, cluster_id):
        return _get_shard_id_from_shiva(cluster_id, self.SHIVA_LOC)

    @staticmethod
    def get_url(url, service_ticket=None):
        args = 'curl -v "%s"' % url

        if service_ticket:
            args += ' -H "X-Ya-Service-Ticket: %s"' % service_ticket

        log.info('Local execute %r', args)
        cmd = subprocess.Popen(
            args,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            shell=True,
        )
        (out, err) = cmd.communicate()
        if cmd.returncode != 0:
            log.warning(
                'Command %r exit with %r: error: %s',
                args, cmd.returncode, err)
        return out

    def run_psql(self, xdb_address, uid=None, sql=None):
        psql = make_psql_command(xdb_address.dsn(self.psql_user, self.psql_password), uid, sql),
        subprocess.Popen(psql, shell=True).communicate()


class Load(Testing):
    SHARPEI = 'http://sharpei-load01f.mail.yandex.net'


ENVIRONMENTS = dict(
    prod=Prod,
    testing=Testing,
    load=Load,
    corp=Corp,
)


def environments_types():
    return ENVIRONMENTS.keys()


class UnconfiguredError(RuntimeError):
    pass


def make_env(env_type, psql_user, arcadia_root, config):
    Env = ENVIRONMENTS[env_type]
    config_items = dict(config.items(env_type))
    if psql_user:
        config_items['psql_user'] = psql_user
    if arcadia_root is None:
        raise Exception('cannot select arcadia_root, set it manually')
    try:
        return Env(env_type=env_type, arcadia_root=arcadia_root, **config_items)
    except TypeError as exc:
        raise UnconfiguredError(str(exc))
