# coding: utf-8

from collections import namedtuple
import json


class Address(namedtuple('Address', ['host', 'port', 'dbname', 'dataCenter'])):
    def dsn(self, user, password):
        return 'postgresql://{user}:{password}@{address.host}:{address.port}/{address.dbname}'.format(
            user=user,
            password=password,
            address=self
        )


class DataBase(namedtuple('DB', ['address', 'role', 'status', 'state'])):
    @staticmethod
    def from_json(desc):
        return DataBase(
            address=Address(**desc.pop('address')),
            **desc
        )


class Shard(namedtuple('Shard', ['id', 'name', 'databases'])):
    @staticmethod
    def from_json(desc):
        return Shard(
            id=desc['id'],
            name=desc['name'],
            databases=[
                DataBase.from_json(dd)
                for dd in desc['databases']
            ]
        )

    @property
    def master(self):
        masters = [d for d in self.databases if d.role == 'master']
        assert len(masters) == 1, \
            'Got unexpected masters count %r' % masters
        return masters[0]

    @property
    def replicas(self):
        replicas = [d for d in self.databases if d.role == 'replica']
        return replicas

    @property
    def replica(self):
        return min(
            (host for host in self.replicas if host.status == 'alive'),
            key=lambda host: host.state['lag']
        )


def parse_conninfo(json_data):
    return Shard.from_json(json.loads(json_data))


def parse_stats(stats_json):
    return [
        Shard.from_json(desc)
        for desc in json.loads(stats_json).values()
    ]


def conninfo_url(host, uid):
    return '{host}/conninfo?mode=all&uid={uid}&format=json'.format(
        host=host,
        uid=uid
    )


def stats_url(host):
    return host + '/v2/stat'
