# coding: utf-8
from argparse import ArgumentParser
import logging
import sys
import os
import json
from pygments import highlight
from pygments.lexers import get_lexer_by_name
from pygments.formatters import get_formatter_by_name
from library.python.find_root import detect_root, is_root


log = logging.getLogger(__name__)


def init_logging(verbosity):
    logging.basicConfig(level=max(logging.DEBUG, logging.WARNING - 10 * verbosity))


def get_query(query_file):
    log.debug("Want query from %r", query_file)
    if query_file:
        if os.path.exists(query_file):
            with open(query_file) as fd:
                return fd.read().strip()
        return query_file


def make_colored(text, lexer_name):
    lexer = get_lexer_by_name(lexer_name)
    formatter = get_formatter_by_name('256', style='native')
    return highlight(text, lexer, formatter)


def pretty_json(json_text):
    text = json.dumps(
        json.loads(json_text),
        sort_keys=True,
        indent=4,
        separators=(',', ': '))
    return make_colored(text, 'json')


def common_arguments():
    from .environment import environments_types

    parser = ArgumentParser(add_help=False)
    parser.add_argument(
        '-e', '--env',
        choices=environments_types(),
        help='environment',
        default='prod',
    )
    parser.add_argument(
        '--db-user',
        metavar='USER',
        dest='db_user',
        help='database user',
    )
    parser.add_argument(
        '-v',
        action='count',
        default=0,
        help='be verbose',
        dest='verbosity',
    )
    parser.add_argument(
        '-a', '--arcadia-root',
        help='specify your arcadia root',
        default=detect_root(
            path=os.path.dirname(sys.path[0]),
            detector=lambda p: is_root(p) or os.path.exists(os.path.join(p, "ya")))
    )
    return parser
