import os
from typing import ClassVar, Optional

from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import padding
from cryptography.hazmat.primitives.ciphers import Cipher, algorithms, modes


class BlockEncryptor:
    """
    AES-CBC-PSK7PADDING encryptor.
    Размер ключа - 128/192/256.
    Размер блока - 128.
    """

    BLOCK_SIZE_IN_BYTES: ClassVar[int] = 16

    @classmethod
    def _generate_iv(cls) -> bytes:
        return os.urandom(cls.BLOCK_SIZE_IN_BYTES)

    @property
    def iv(self) -> bytes:
        return self._iv

    def __init__(self, key: bytes, iv: Optional[bytes] = None) -> None:
        if iv is None:
            iv = self._generate_iv()

        self._iv = iv
        assert len(self._iv) == self.BLOCK_SIZE_IN_BYTES

        self._cipher = Cipher(algorithms.AES(key), modes.CBC(iv), backend=default_backend())
        self._encryptor = self._cipher.encryptor()
        self._padder = padding.PKCS7(self.BLOCK_SIZE_IN_BYTES * 8).padder()

    def update(self, data: bytes) -> bytes:
        return self._encryptor.update(self._padder.update(data))

    def finalize(self) -> bytes:
        data = self._encryptor.update(self._padder.finalize())
        return data + self._encryptor.finalize()


class BlockDecryptor:
    """
    AES-CBC-PSK7PADDING decryptor.
    Размер ключа - 128/192/256.
    Размер блока - 128.
    """

    BLOCK_SIZE_IN_BYTES: ClassVar[int] = 16

    def __init__(self, key: bytes, iv: bytes) -> None:
        assert len(iv) == self.BLOCK_SIZE_IN_BYTES

        self._cipher = Cipher(algorithms.AES(key), modes.CBC(iv), backend=default_backend())
        self._decryptor = self._cipher.decryptor()
        self._unpadder = padding.PKCS7(self.BLOCK_SIZE_IN_BYTES * 8).unpadder()

    def update(self, data: bytes) -> bytes:
        return self._unpadder.update(self._decryptor.update(data))

    def finalize(self) -> bytes:
        data = self._unpadder.update(self._decryptor.finalize())
        return data + self._unpadder.finalize()
