import csv
from typing import Optional, Type


def sniff(sample: str, delimiters: str = ";,\t") -> Type[csv.Dialect]:
    if not sample:
        return csv.excel

    dialect = _guess_first_line_dialect(sample, delimiters)
    if not dialect:
        try:
            dialect = csv.Sniffer().sniff(sample, delimiters=delimiters)
        except csv.Error:
            return csv.excel

    return _fix_dialect_quoting(dialect)


def _guess_first_line_dialect(sample: str, delimiters: str) -> Optional[Type[csv.Dialect]]:
    first_line = sample.splitlines()[0]
    try:
        return csv.Sniffer().sniff(first_line, delimiters=delimiters)
    except csv.Error:
        return None


def _fix_dialect_quoting(dialect: Type[csv.Dialect]) -> Type[csv.Dialect]:
    if not dialect.doublequote and not dialect.escapechar:
        dialect.doublequote = True
    return dialect
