import sys
import logging
import logging.handlers
from dataclasses import dataclass
from .deploy_formatter import DeployFormatter


def _setup_logger(logger):
    logger.propagate = False
    logger.level = logging.INFO
    handler = logging.StreamHandler(sys.stdout)
    handler.formatter = DeployFormatter()
    logger.addHandler(handler)


_logger = logging.getLogger("global")
_setup_logger(_logger)


def _setup_file_handler(file_path):
    file_handler = logging.handlers.WatchedFileHandler(file_path)
    file_handler.formatter = DeployFormatter()
    memory_handler = logging.handlers.MemoryHandler(
        capacity=1024, flushLevel=logging.INFO, target=file_handler
    )
    _logger.addHandler(memory_handler)


def setup_logger(settings):
    if settings.file:
        _setup_file_handler(settings.file)


@dataclass
class DeployLogger:
    logger: logging.Logger

    def set_additional_fields(self, fields):
        self.additional_fields = fields
        return self

    def log(self, level, message, **kwargs):
        self.logger.log(
            level, message, extra=self._prepare_entries(**{**kwargs, **self.additional_fields})
        )

    def debug(self, message, **kwargs):
        self.log(logging.DEBUG, message, **kwargs)

    def info(self, message, **kwargs):
        self.log(logging.INFO, message, **kwargs)

    def error(self, message, **kwargs):
        self.log(logging.ERROR, message, **kwargs)

    def exception(self, message, **kwargs):
        self.logger.exception(
            message, extra=self._prepare_entries(**{**kwargs, **self.additional_fields})
        )

    def _prepare_entries(self, **kwargs):
        return {k: prepare_value(v) for k, v in kwargs.items()}


def prepare_value(value):
    return str(value).replace("\n", "\\n").replace("\r", "\\r").replace("\t", "\\t")


def get_logger(**kwargs):
    return DeployLogger(_logger).set_additional_fields(kwargs)
