import json
from datetime import timedelta
from typing import Any, Dict, List, Optional
from psycopg2.extensions import cursor
from tractor.db import BaseDatabase
from tractor.mail.models import (
    MessagesCollectionStats,
    UserMigration,
    UserMigrationStatus,
    UserMigrationExportInfo,
    UserMigrationInfo,
)
from tractor.settings import TractorDBSettings


class Database(BaseDatabase):
    def __init__(self, settings: TractorDBSettings):
        super().__init__(settings, "tractor_mail")

    def create_user_migration(
        self,
        org_id: str,
        domain: str,
        login: str,
        prepare_task_id: int,
        cur: cursor,
    ):
        cur.execute(
            """
                INSERT INTO tractor_mail.user_migrations (
                    org_id,
                    login,
                    domain,
                    status,
                    prepare_task_id
                )
                VALUES (
                    %(org_id)s::text,
                    %(login)s::text,
                    %(domain)s::text,
                    %(status)s::tractor_mail.user_migration_status,
                    %(prepare_task_id)s::bigint
                )
            """,
            {
                "org_id": org_id,
                "login": login,
                "domain": domain,
                "status": "preparing",
                "prepare_task_id": prepare_task_id,
            },
        )

    def get_user_migration(self, org_id: str, login: str, cur: cursor):
        cur.execute(
            """
                SELECT
                    org_id,
                    login,
                    domain,
                    status,
                    error_reason,
                    stats,
                    prepare_task_id,
                    stop_task_id
                FROM tractor_mail.user_migrations
                WHERE
                    org_id = %(org_id)s::text
                    AND
                    login = %(login)s::text
                LIMIT 1
            """,
            {"org_id": org_id, "login": login},
        )
        res = cur.fetchone()
        if res is None:
            return None
        user_migration = self._build_migration_object_from_db_response(res)
        return user_migration

    def reset_user_migration(self, org_id: str, login: str, domain: str, prepare_task_id: int, cur):
        cur.execute(
            """
                UPDATE tractor_mail.user_migrations
                SET
                    status = 'preparing'::tractor_mail.user_migration_status,
                    error_reason = '',
                    stats = '{}'::jsonb,
                    stats_touch_ts = NULL,
                    prepare_task_id = %(prepare_task_id)s::bigint,
                    stop_task_id = NULL,
                    domain = %(domain)s::text
                WHERE
                    org_id = %(org_id)s::text
                    AND
                    login = %(login)s::text
            """,
            {
                "org_id": org_id,
                "login": login,
                "prepare_task_id": prepare_task_id,
                "domain": domain,
            },
        )

    def acquire_and_stop_migrations(self, org_id: str, cur: cursor) -> List[Dict[str, str]]:
        cur.execute(
            """
                WITH acquired_migrations AS (
                    SELECT
                        org_id,
                        login,
                        status,
                        prepare_task_id
                    FROM tractor_mail.user_migrations
                    WHERE
                    org_id = %(org_id)s::text
                    AND
                    (
                        status = 'preparing'
                        OR
                        status = 'initial_sync'
                        OR
                        status = 'sync_newest'
                    )
                    FOR UPDATE
                ),
                -- stopped_migrations variable needed just to fit all the logic into one query
                stopped_migrations AS (
                    UPDATE tractor_mail.user_migrations
                    SET
                        status='stopping'
                    WHERE
                        (org_id, login) IN (
                            SELECT
                                org_id,
                                login
                            FROM acquired_migrations
                        )
                )
                SELECT
                    prepare_task_id,
                    login,
                    status
                FROM
                    acquired_migrations
            """,
            {"org_id": org_id},
        )
        return [
            {
                "prepare_task_id": tup[0],
                "login": tup[1],
                "previous_status": tup[2],
            }
            for tup in cur.fetchall()
        ]

    def cancel_task(self, task_id: str, cur: cursor):
        cur.execute(
            """
            UPDATE tractor_mail.tasks
            SET
                canceled = True
            WHERE
                task_id = %(task_id)s
                AND
                canceled = False
                AND
                worker_status = 'pending'
            """,
            {
                "task_id": task_id,
            },
        )

    def attach_stop_task_to_migration(
        self,
        stop_task_id: str,
        org_id: str,
        login: str,
        cur: cursor,
    ):
        cur.execute(
            """
                UPDATE tractor_mail.user_migrations
                SET
                    stop_task_id = %(stop_task_id)s
                WHERE
                    org_id = %(org_id)s::text
                    AND
                    login = %(login)s::text
            """,
            {
                "org_id": org_id,
                "login": login,
                "stop_task_id": stop_task_id,
            },
        )

    def get_user_migration_statuses(self, org_id: str, cur: cursor) -> List[UserMigrationInfo]:
        cur.execute(
            """
                SELECT
                    login,
                    status,
                    error_reason
                FROM tractor_mail.user_migrations
                WHERE
                    org_id = %(org_id)s::text
            """,
            {"org_id": org_id},
        )
        return [
            UserMigrationInfo(
                login=record[0],
                status=UserMigrationStatus(record[1]),
                error_reason=record[2],
            )
            for record in cur
        ]

    def export_user_migrations(self, org_id: str, cur: cursor) -> List[UserMigrationExportInfo]:
        cur.execute(
            """
            SELECT login, status, error_reason, stats
            FROM tractor_mail.user_migrations
            WHERE org_id = %(org_id)s::text
            """,
            {"org_id": org_id},
        )
        return [_make_export_info(*record) for record in cur]

    def acquire_migration_for_stats_update(
        self,
        update_interval: timedelta,
        cur: cursor,
    ) -> Optional[UserMigration]:
        cur.execute(
            """
                WITH selected AS (
                    SELECT
                        org_id,
                        login
                    FROM tractor_mail.user_migrations
                    WHERE
                        (
                            status = 'initial_sync'
                            OR
                            status = 'sync_newest'
                        )
                        AND
                        (
                            stats_touch_ts IS NULL
                            OR
                            NOW() - stats_touch_ts >= %(update_interval)s
                        )
                    ORDER BY stats_touch_ts
                    LIMIT 1
                    FOR UPDATE SKIP LOCKED
                )
                UPDATE tractor_mail.user_migrations
                SET
                    stats_touch_ts = NOW()
                FROM selected
                WHERE
                    tractor_mail.user_migrations.org_id = selected.org_id
                    AND
                    tractor_mail.user_migrations.login = selected.login
                RETURNING
                    selected.org_id,
                    selected.login,
                    domain,
                    status,
                    error_reason,
                    stats,
                    prepare_task_id,
                    stop_task_id
            """,
            {
                "update_interval": update_interval,
            },
        )
        res = cur.fetchone()
        if res is None:
            return None
        else:
            return self._build_migration_object_from_db_response(res)

    def update_stats(self, org_id: str, login: str, stats: dict, cur: cursor):
        stats_str = json.dumps(stats)
        cur.execute(
            """
                UPDATE tractor_mail.user_migrations
                SET
                    stats = %(stats)s::jsonb
                WHERE
                    org_id = %(org_id)s::text
                    AND
                    login = %(login)s::text
            """,
            {
                "org_id": org_id,
                "login": login,
                "stats": stats_str,
            },
        )

    def move_migration_to_new_status(
        self,
        org_id: str,
        login: str,
        current_status: UserMigrationStatus,
        new_status: UserMigrationStatus,
        cur: cursor,
        error_reason="",
    ):
        cur.execute(
            """
                UPDATE tractor_mail.user_migrations
                SET
                    status = %(new_status)s,
                    error_reason = %(error_reason)s
                WHERE
                    org_id = %(org_id)s::text
                    AND
                    login = %(login)s::text
                    AND
                    status = %(current_status)s
            """,
            {
                "org_id": org_id,
                "login": login,
                "current_status": current_status.value,
                "new_status": new_status.value,
                "error_reason": error_reason,
            },
        )

    def mark_migration_finished(
        self,
        org_id: str,
        login: str,
        status: UserMigrationStatus,
        error_reason: str,
        cur: cursor,
    ):
        cur.execute(
            """
                UPDATE tractor_mail.user_migrations
                SET
                    status = %(status)s::tractor_mail.user_migration_status,
                    error_reason = %(error_reason)s::text
                WHERE
                    org_id = %(org_id)s::text
                    AND
                    login = %(login)s::text
            """,
            {
                "org_id": org_id,
                "login": login,
                "status": status.value,
                "error_reason": error_reason,
            },
        )

    def acquire_migration_in_preparing_status(self, cur: cursor) -> Optional[UserMigration]:
        cur.execute(
            """
                SELECT
                    m.org_id,
                    m.login,
                    m.domain,
                    m.status,
                    m.error_reason,
                    m.stats,
                    m.prepare_task_id,
                    m.stop_task_id
                FROM
                    tractor_mail.user_migrations AS m
                    INNER JOIN
                    tractor_mail.tasks AS t
                    ON
                    m.prepare_task_id = t.task_id
                WHERE
                    m.status = 'preparing'
                    AND
                    t.worker_status != 'pending'
                FOR UPDATE SKIP LOCKED
                LIMIT 1
            """
        )
        res = cur.fetchone()
        if res is None:
            return None
        else:
            return self._build_migration_object_from_db_response(res)

    def acquire_migration_in_stopping_status(self, cur: cursor) -> Optional[UserMigration]:
        cur.execute(
            """
                SELECT
                    m.org_id,
                    m.login,
                    m.domain,
                    m.status,
                    m.error_reason,
                    m.stats,
                    m.prepare_task_id,
                    m.stop_task_id
                FROM
                    tractor_mail.user_migrations as m
                    INNER JOIN
                    tractor_mail.tasks AS t
                    ON
                    m.stop_task_id = t.task_id
                WHERE
                    status = 'stopping'
                    AND
                    t.worker_status != 'pending'
                FOR UPDATE SKIP LOCKED
                LIMIT 1
            """
        )
        res = cur.fetchone()
        if res is None:
            return None
        else:
            return self._build_migration_object_from_db_response(res)

    def _build_migration_object_from_db_response(self, resp: tuple) -> UserMigration:
        migration = UserMigration(
            org_id=resp[0],
            login=resp[1],
            domain=resp[2],
            status=UserMigrationStatus(resp[3]),
            error_reason=resp[4],
            stats=resp[5],
            prepare_task_id=resp[6],
            stop_task_id=resp[7],
        )
        return migration


def _make_export_info(
    login: str,
    status: str,
    error_reason: str,
    stats: Dict[str, Any],
) -> UserMigrationExportInfo:
    folders_json: Optional[Dict[str, Dict[str, int]]] = stats.get("folders")
    if folders_json is None:
        messages_collection_stats = None
    else:
        messages_collection_stats = MessagesCollectionStats()
        for folder in folders_json.values():
            messages_collection_stats.source_count += folder["src_mailbox_messages_count"]
            messages_collection_stats.collected_count += folder["collected_messages_count"]
            messages_collection_stats.failed_count += folder["failed_to_collect_messages_count"]
    return UserMigrationExportInfo(
        login=login,
        status=UserMigrationStatus(status),
        error_reason=error_reason,
        mailbox_messages_collection_stats=messages_collection_stats,
    )
