from dataclasses import dataclass
from typing import Dict, Any, List
import os
import yaml


_RawSettings = Dict[str, Any]


class Settings:
    HIERARCHY_SEPARATOR = "/"

    def __init__(self) -> None:
        self.data = self._load_config()

    def get(self, path: str) -> Any:
        key_parts = path.split(Settings.HIERARCHY_SEPARATOR)
        return self._get_value_using_key_parts(self.data, key_parts)

    def get_or_default(self, path: str, default: Any) -> Any:
        key_parts = path.split(Settings.HIERARCHY_SEPARATOR)
        val = self._get_value_using_key_parts(self.data, key_parts)
        if val is None:
            return default
        return val

    def _get_value_using_key_parts(self, data: _RawSettings, parts: List[str]) -> Any:
        top_level_key = parts[0]
        if top_level_key not in data:
            return None

        value = data[top_level_key]
        if len(parts) == 1:
            return value
        else:
            return self._get_value_using_key_parts(value, parts[1:])

    def _load_config(self) -> _RawSettings:
        config_path = os.environ.get("CONFIG_PATH")
        if config_path is None:
            raise RuntimeError('env variable "CONFIG_PATH" not defined')
        config = self._load_config_file(config_path)
        _resolve_env_variables(config)
        return config

    def _load_config_file(self, filename: str) -> _RawSettings:
        with open(filename) as config_file:
            config_data = yaml.safe_load(config_file)
            if "extends" not in config_data:
                return config_data
            parent_config = self._load_config_file(config_data["extends"])
            return _merge_configs(parent_config, config_data)


def _resolve_env_variables(config: _RawSettings) -> None:
    updates = {}
    for key, val in config.items():
        if isinstance(val, dict):
            _resolve_env_variables(val)
        elif isinstance(val, str) and val.startswith("$"):
            updates[key] = os.environ[val[1:]]

    config.update(updates)


def _merge_configs(parent: _RawSettings, child: _RawSettings) -> _RawSettings:
    merged = dict()

    for key, parent_val in parent.items():
        if key in child:
            child_val = child[key]
            if type(parent_val) != type(child_val):
                raise Exception(
                    f"Bad override for {key} in child config(parent type is {type(parent_val)}, child type is {type(child_val)}"
                )

            if type(parent_val) is dict:
                merged[key] = _merge_configs(parent_val, child_val)
            elif type(parent_val) is list:
                merged[key] = parent_val + child_val
            else:
                merged[key] = child_val
        else:
            merged[key] = parent_val

    for key, child_val in child.items():
        if key not in parent:
            merged[key] = child_val

    return merged


@dataclass
class RetryingSettings:
    count: int


@dataclass
class RetryingSettingsWithDelay(RetryingSettings):
    base_delay_in_seconds: float


@dataclass
class DirectorySettings:
    host: str
    timeout: int = 5
    pagination_size: int = 1000
    user_agent: str = "tractor_disk"


@dataclass
class BlackboxSettings:
    host: str
    timeout: int = 5
    user_agent: str = "tractor_disk"
    retrying: RetryingSettingsWithDelay = RetryingSettingsWithDelay(
        base_delay_in_seconds=0.100, count=3
    )


@dataclass
class TractorDBSettings:
    conninfo: str
