import dataclasses
import json
from datetime import datetime
from enum import Enum
from typing import Any, Dict, Set, Type, Union, get_origin, get_args


def name_of_field(field: dataclasses.Field) -> str:
    return field.name


def construct_from_dict(dataclass: Type, data: Dict) -> Any:
    field_types = get_field_types(dataclass)
    args = {}
    for k, v in data.items():
        if issubclass(field_types[k], Enum):
            args[k] = field_types[k][v.lower()]
        elif dataclasses.is_dataclass(field_types[k]) and type(v) is dict:
            args[k] = construct_from_dict(field_types[k], v)
        else:
            args[k] = field_types[k](v)
    return dataclass(**args)


def construct_from_dict_db(dataclass: Type, data: Dict) -> Any:
    field_types = get_field_types(dataclass)
    args = {}
    data = {k: v for k, v in data.items() if k in field_types}
    for k, v in data.items():
        if issubclass(field_types[k], Enum):
            args[k] = field_types[k](v.lower())
        elif dataclasses.is_dataclass(field_types[k]) and type(v) is dict:
            args[k] = construct_from_dict(field_types[k], v)
        elif field_types[k] is datetime and (v is None or type(v) is datetime):
            args[k] = v
        else:
            args[k] = field_types[k](v)
    return dataclass(**args)


def get_field_list(dataclass: Type) -> Set[str]:
    return {f.name for f in dataclasses.fields(dataclass)}


def get_optional_field_list(dataclass: Type) -> Set[str]:
    return {f.name for f in dataclasses.fields(dataclass) if _has_default(f)}


def get_field_types(dataclass: Type) -> Dict[str, Type]:
    return {f.name: _get_instantination_type(f.type) for f in dataclasses.fields(dataclass)}


def _has_default(field: dataclasses.Field) -> bool:
    return (
        field.default is not dataclasses.MISSING or field.default_factory is not dataclasses.MISSING  # type: ignore
    )


def _get_instantination_type(field_type: Type) -> Type:
    origin_type = get_origin(field_type)
    if origin_type is None:
        return field_type

    if origin_type is not Union:
        raise Exception(f"unsupported type for direct instantination: {field_type}")

    union_args = get_args(field_type)
    if not union_args:
        raise Exception(f"broken Union: {field_type}")

    if len(union_args) == 1:
        return union_args[0]

    if len(union_args) > 2:
        raise Exception(f"ambigous type for instantination: {field_type}")

    if type(None) in union_args:
        none_index = union_args.index(type(None))
        return union_args[none_index - 1]

    raise Exception(f"ambigous type for instantination: {field_type}")


def transform_into_json_str_without_none(obj):
    d = {}
    for field, value in vars(obj).items():
        if value is None:
            continue
        d[field] = value
    return json.dumps(d)
