from dataclasses import dataclass
from tractor.util.retrying import TemporaryError, retry
from tractor.secrets import Secrets
from urllib.parse import urlencode
import requests

ORG_IDS_ATTR = "1017"  # account.connect_organization_ids


class Blackbox:
    def __init__(self, blackbox_settings):
        self.settings = blackbox_settings

    @dataclass
    class DBFieldsMapping:
        suid: str = "subscription.suid.2"

    def userinfo(self, uid):
        res = self._do_request(
            method="userinfo", uid=uid, userip="127.0.0.1", emails="getall", attributes=ORG_IDS_ATTR
        )
        if len(res["users"]) == 0:
            raise RuntimeError("user not found")
        user = res["users"][0]
        if not user["uid"]:
            raise RuntimeError("user not found")
        return user

    def get_suid(self, uid):
        res = self._do_request(
            method="userinfo", uid=uid, userip="127.0.0.1", dbfields=Blackbox.DBFieldsMapping.suid
        )
        if len(res["users"]) == 0:
            raise RuntimeError("user not found")
        user = res["users"][0]
        return user["dbfields"][Blackbox.DBFieldsMapping.suid]

    def _do_request(self, **kwargs):
        kwargs["format"] = "json"
        headers = {
            "User-Agent": self.settings.user_agent,
            "X-Ya-Service-Ticket": Secrets().blackbox_secret(),
        }

        @retry(
            base_delay=self.settings.retrying.base_delay_in_seconds,
            retries=self.settings.retrying.count,
        )
        def impl():
            resp = requests.get(
                f"{self.settings.host}/blackbox?{urlencode(kwargs)}",
                headers=headers,
                timeout=self.settings.timeout,
            )
            resp.raise_for_status()
            resp_data = resp.json()
            if "exception" in resp_data:
                error_code = resp_data["exception"]["value"]
                message = "blackbox error {} {}".format(error_code, resp_data["error"])
                if (
                    error_code == "DB_EXCEPTION"
                ):  # See https://docs.yandex-team.ru/blackbox/concepts/blackboxErrors#retries-policy.
                    raise TemporaryError(message)
                raise RuntimeError(message)
            return resp_data

        return impl()
