from dataclasses import dataclass
from typing import Dict, List
import requests
from enum import Enum
from tractor.mail.models import MailServerConnectionInfo, MessagesCollectionStats
from tractor.secrets import Secrets


class CollectorsException(Exception):
    def __init__(self, reason: str, descrintion: str):
        super().__init__()
        self.reason = reason
        self.description = descrintion

    def __str__(self) -> str:
        return f"Collectors error: reason={self.reason}, description={self.description}"


@dataclass
class CollectorInfo:
    bad_retries: int
    error_status: str
    is_oauth: bool
    is_on: int
    last_connect: str
    last_msg_count: int
    login: str
    popid: str
    server_info: MailServerConnectionInfo


class Collectors:
    class FixedParameters(Enum):
        mdb = "pg"
        json = "1"
        IMAP = "1"

    def __init__(self, settings) -> None:
        self.settings = settings

    def list(self, uid: str, suid: str) -> List[CollectorInfo]:
        response = self._do_request(
            f"{self.settings.host}/api/list",
            params={"mdb": Collectors.FixedParameters.mdb.value, "uid": uid, "suid": suid},
        )
        collectors_dct_list: List[Dict[str, str]] = response["rpops"]
        result = []
        for collector in collectors_dct_list:
            result.append(
                CollectorInfo(
                    int(collector["bad_retries"]),
                    collector["error_status"],
                    bool(collector["is_oauth"]),
                    int(collector["is_on"]),
                    collector["last_connect"],
                    collector["last_msg_count"],
                    collector["login"],
                    collector["popid"],
                    MailServerConnectionInfo(
                        collector["server"],
                        int(collector["port"]),
                        bool(collector["use_ssl"]),
                    ),
                )
            )
        return result

    def status(self, uid: str, suid: str, popid: str) -> Dict[str, MessagesCollectionStats]:
        response = self._do_request(
            f"{self.settings.host}/api/status",
            params={
                "mdb": Collectors.FixedParameters.mdb.value,
                "uid": uid,
                "suid": suid,
                "popid": popid,
            },
        )
        ret = {}
        for folder in response["folders"]:
            folder_stats = MessagesCollectionStats()
            folder_stats.source_count = int(folder["messages"])
            folder_stats.collected_count = int(folder["collected"])
            folder_stats.failed_count = int(folder["errors"])
            ret[folder["name"]] = folder_stats
        return ret

    def create(
        self,
        email: str,
        password: str,
        uid,
        suid,
        mail_server_info: MailServerConnectionInfo = None,
    ) -> str:
        params = {
            "login": email,
            "password": password,
            "uid": uid,
            "suid": suid,
            "mdb": Collectors.FixedParameters.mdb.value,
            "user": email,
        }
        if mail_server_info is not None:
            params.update(
                {
                    "imap": Collectors.FixedParameters.IMAP.value,
                    "port": mail_server_info.port,
                    "ssl": mail_server_info.ssl,
                    "server": mail_server_info.host,
                }
            )
        response = self._do_request(f"{self.settings.host}/api/create", params=params)
        return response["popid"]

    def delete(self, uid: str, suid: str, popid: str):
        response = self._do_request(
            url=f"{self.settings.host}/api/delete",
            params={
                "uid": uid,
                "suid": suid,
                "popid": popid,
                "mdb": Collectors.FixedParameters.mdb.value,
            },
        )

    def _do_request(self, url: str, params: Dict[str, str]={}, headers: Dict[str, str]={}, method="GET"):
        params.update({"json": Collectors.FixedParameters.json.value})
        headers.update(
            {
                "User-Agent": self.settings.user_agent,
                "X-Ya-Service-Ticket": Secrets().collectors_secret(),
            }
        )
        response = requests.request(
            url=url,
            params=params,
            headers=headers,
            method=method,
            timeout=self.settings.timeout,
        )
        response.raise_for_status()
        response = response.json()
        self._validate_response(response)
        return response

    def _validate_response(self, response: Dict[str, str]):
        if "error" not in response:
            return
        reason = response["error"]["reason"]
        description = response["error"]["description"]
        raise CollectorsException(reason, description)
