from datetime import datetime
import re
import requests
import urllib
from tractor.secrets import Secrets
from tractor.mail.models import UserInfo


class Directory:
    USER_BIRTHDAY_FORMAT = "%d.%m.%Y"
    ALLOWED_LANGUAGES = {"en", "ru"}
    ALLOWED_GENDERS = {"male", "female"}

    def __init__(self, orgid, settings):
        self.orgid = orgid
        self.settings = settings

    def get_users(self):
        ret = self._do_users_request(self.orgid)
        return ret

    def get_domain(self) -> str:
        resp = self._do_organizations_request()
        if "domains" not in resp:
            raise Exception('no "domains" field in Directory response')
        domain = resp["domains"].get("master")
        assert isinstance(domain, str), type(domain)
        if _contains_cyrillic(domain):
            domain = _to_punycode(domain)
        return domain

    def create_user(self, admin_uid, user_info: UserInfo):
        url = f"{self.settings.host}/v11/users"
        headers = self._common_headers()
        headers.update(
            {
                "X-ORG-ID": str(self.orgid),
                "X-UID": str(admin_uid),
                "X-User-IP": "127.0.0.1",
            }
        )
        req_data = self._make_create_user_request_data(user_info)
        resp = self._do_post_request(url, headers, data=req_data)
        uid = resp["id"]
        return uid

    def _make_create_user_request_data(self, user_info: UserInfo):
        data = {
            "nickname": user_info.login,
            "department_id": 1,
            "password": user_info.yandex_password,
            "password_mode": "plain",
        }
        name: dict = {}
        if user_info.first_name is not None:
            name.update(
                {
                    "first": {
                        "ru": user_info.first_name,
                        "en": user_info.first_name,
                    }
                }
            )
        if user_info.last_name is not None:
            name.update(
                {
                    "last": {
                        "ru": user_info.last_name,
                        "en": user_info.last_name,
                    }
                }
            )
        if user_info.middle_name is not None:
            name.update(
                {
                    "middle": {
                        "ru": user_info.middle_name,
                        "en": user_info.middle_name,
                    }
                }
            )
        if name:
            data["name"] = name
        if user_info.gender is not None:
            data["gender"] = user_info.gender
        if user_info.birthday is not None:
            birthday = datetime.strptime(user_info.birthday, Directory.USER_BIRTHDAY_FORMAT).date()
            data["birthday"] = birthday.isoformat()
        if user_info.language is not None:
            data["language"] = user_info.language
        return data

    def _do_users_request(self, orgid):
        url = "{}/v11/users/?".format(self.settings.host) + urllib.parse.urlencode(
            {
                "fields": "email",
                "per_page": self.settings.pagination_size,
            }
        )
        headers = self._common_headers()
        headers.update({"X-ORG-ID": str(orgid)})

        ret = []
        while True:
            chunk = self._do_get_request(url, headers)
            self._validate_users_response(chunk)
            ret += chunk["result"]
            if ("links" not in chunk) or ("next" not in chunk["links"]):
                break
            url = chunk["links"]["next"]

        for user in ret:
            user["uid"] = str(user["id"])
            del user["id"]

        return ret

    def _do_organizations_request(self):
        url = "{}/v11/organizations/{}/?".format(
            self.settings.host, self.orgid
        ) + urllib.parse.urlencode(
            {
                "fields": "domains",
            }
        )
        headers = self._common_headers()
        return self._do_get_request(url, headers)

    def _do_get_request(self, url, headers={}):
        resp = requests.get(url, headers=headers, timeout=self.settings.timeout)
        resp.raise_for_status()
        return resp.json()

    def _do_post_request(self, url, headers = {}, data={}):
        resp = requests.post(url, headers=headers, json=data, timeout=self.settings.timeout)
        resp.raise_for_status()
        return resp.json()

    def _common_headers(self):
        return {
            "User-Agent": self.settings.user_agent,
            "X-Ya-Service-Ticket": Secrets().directory_secret(),
        }

    def _validate_users_response(self, resp_json):
        if "result" not in resp_json:
            raise Exception('no "result" field in Directory response')
        if not isinstance(resp_json["result"], list):
            raise Exception('"result" field in Directory response is not list')
        for user_json in resp_json["result"]:
            if "id" not in user_json:
                raise Exception('no "id" field for user in Directory response')
            if not isinstance(user_json["id"], int) and not isinstance(user_json["id"], unicode):
                raise Exception(
                    '"id" field of user in Directory response is not int and not unicode'
                )


def _contains_cyrillic(string):
    return bool(re.search("[а-яА-Я]", string))


def _to_punycode(string: str) -> str:
    return string.encode("idna").decode("utf-8")
