from dataclasses import dataclass
from flask import Response
from tractor_api.app.access_log import add_fields_to_log
from tractor_api.app.responses import make_error_response
from typing import Optional, Dict, Any
from werkzeug.exceptions import HTTPException, InternalServerError
import traceback


@dataclass
class AppException(Exception):
    error_code: str = "internal_error"
    status_code: Optional[int] = 500
    detail: Optional[Dict[str, Any]] = None


def handle_http_error(e: HTTPException) -> Response:
    _add_exception_to_log(e)
    return _make_error_response(AppException(e.name, e.code))


def handle_app_exception(e: AppException) -> Response:
    _add_exception_to_log(e)
    return _make_error_response(e)


def handle_unhandled_exception(e: InternalServerError) -> Response:
    _add_exception_to_log(e.original_exception)  # type: ignore
    return _make_error_response(AppException("internal_error", 500))


def _add_exception_to_log(e: Exception) -> None:
    add_fields_to_log(reason=str(e))
    stack_summary = traceback.extract_tb(e.__traceback__)
    last_frame = stack_summary[-1]
    add_fields_to_log(
        last_frame=str(f"{last_frame.filename}:{last_frame.lineno} in {last_frame.name}")
    )


def _make_error_response(exception: AppException) -> Response:
    return make_error_response(exception.status_code, exception.error_code, exception.detail or {})
