import json
from flask import request
from tractor.util.dataclasses import get_field_list, get_optional_field_list, construct_from_dict
from tractor_api.app.error_handling import AppException
from typing import Any, List, Optional, Type


class MissingParamsException(AppException):
    def __init__(self, names: List[str]) -> None:
        super(MissingParamsException, self).__init__(
            error_code="missing_params",
            status_code=400,
            detail={"missing_params": names},
        )


class InvalidParamsException(AppException):
    def __init__(self, names: str) -> None:
        super(InvalidParamsException, self).__init__(
            error_code="invalid_params",
            status_code=400,
            detail={"invalid_params": names},
        )


class UnsupportedArgumentException(AppException):
    def __init__(self, name: str) -> None:
        super(UnsupportedArgumentException, self).__init__(
            error_code="unsupported_request",
            status_code=400,
            detail={"unsupported_argument": name},
        )


class ConcurrentOperationException(AppException):
    def __init__(self, name: str) -> None:
        super(ConcurrentOperationException, self).__init__(
            error_code="concurrent_error",
            status_code=409,
            detail={
                "concurrent_operation": name,
            },
        )


def fill_dataclass_using_args(dataclass: Type, argument_prefix: Optional[str] = None) -> Any:
    fields = get_field_list(dataclass)
    optional_fields = get_optional_field_list(dataclass)
    args = {}
    missing_args = []
    for field in fields:
        arg_name = _apply_argument_prefix(field, argument_prefix)
        if arg_name in request.args:
            args[field] = request.args[arg_name]
        elif field not in optional_fields:
            missing_args.append(arg_name)
    if missing_args:
        raise MissingParamsException(missing_args)
    return construct_from_dict(dataclass, args)


def _apply_argument_prefix(name: str, prefix: Optional[str]) -> str:
    if prefix is None:
        return name
    return f"{prefix}{name}"


def get_required_argument(name: str) -> str:
    try:
        return request.args[name]
    except KeyError as exc:
        raise MissingParamsException([name]) from exc


def get_required_arguments(*names: str) -> List[str]:
    missing_arguments = [name for name in names if name not in request.args]
    if missing_arguments:
        raise MissingParamsException(missing_arguments)

    return [request.args[name] for name in names]


def get_request_body_as_json():
    raw_req_body = request.get_data(cache=False)
    try:
        req_body = json.loads(raw_req_body)
    except Exception as e:
        raise InvalidParamsException("request_body")
    return req_body
