from enum import Enum
from flask import Blueprint, request, g, current_app
from functools import wraps
import requests
from tractor_api.settings import settings
from tractor_api.app.error_handling import AppException
from tractor_api.app.access_log import add_fields_to_log
from typing import Any, Callable, Optional
from os import environ


class TicketType(Enum):
    SERVER = {"name": "srv", "header": "X-Ya-Service-Ticket"}
    USER = {"name": "usr", "header": "X-Ya-User-Ticket"}

    def __str__(self) -> str:
        return self.value["name"]

    def header(self) -> str:
        return self.value["header"]


service_tvm = Blueprint("service_tvm", __name__)
user_tvm = Blueprint("user_tvm", __name__)


class TvmAuthException(AppException):
    def __init__(self, description: str, logging_string: str = None):
        super(TvmAuthException, self).__init__(
            error_code="tvm_auth_error", status_code=401, detail={"description": description}
        )
        self.logging_string = logging_string


def allowed_service(*services: str) -> Callable:
    def allowed_service_decorator(func: Callable) -> Callable:
        @wraps(func)
        def wrapper(*args: Any, **kwargs: Any) -> Any:
            if not _is_tvm_enforced():
                return func(*args, **kwargs)
            if g.tvm_service not in services:
                raise TvmAuthException("not allowed tvm source")
            return func(*args, **kwargs)

        return wrapper

    return allowed_service_decorator


@service_tvm.before_request
def check_service_tvm() -> None:
    ticket = _get_ticket(TicketType.SERVER)
    if ticket is None:
        return

    res = _check_ticket(ticket, TicketType.SERVER)
    if res is None:
        return

    src_id = res["src"]
    g.tvm_service = settings().get_or_default(f"tvm/clients/{src_id}", "unknown")
    add_fields_to_log(tvm_src_id=src_id, tvm_src=g.tvm_service)


@user_tvm.before_request
def check_user_tvm() -> None:
    ticket = _get_ticket(TicketType.USER)
    if ticket is None:
        g.uid = request.headers.get("X-UID")
        return

    res = _check_ticket(ticket, TicketType.USER)
    if res is None:
        return

    g.uid = res["default_uid"]
    add_fields_to_log(user_id=g.uid)


def _is_tvm_enforced() -> bool:
    return settings().get_or_default("tvm/enforced", True)


def _get_ticket(ticket_type: TicketType) -> Optional[str]:
    ticket = request.headers.get(ticket_type.header())
    if _is_tvm_enforced() and ticket is None:
        raise TvmAuthException(f"no {ticket_type} ticket")
    return ticket


def _check_ticket(ticket: str, ticket_type: TicketType) -> Any:
    resp = _tvmtool_check(ticket, ticket_type)
    if resp is None:
        return resp

    if "error" in resp:
        logging_string = f"error={resp['error']}"
        if _is_tvm_enforced():
            raise TvmAuthException(f"bad {ticket_type} ticket", logging_string)
        else:
            resp = None

    return resp


def _tvmtool_check(ticket: str, ticket_type: TicketType) -> Any:
    try:
        tvmtool_url = _get_deploy_tvm_url()
        auth_token = _get_auth_token()
        response = requests.get(
            f"{tvmtool_url}/tvm/check{ticket_type}",
            params={"dst": "tractor"},
            headers={"Authorization": auth_token, ticket_type.header(): ticket},
        )
        return response.json()
    except Exception as e:
        current_app.logger.error("check ticket exception: %s", str(e))
        if _is_tvm_enforced():
            raise
        else:
            return None


def _get_auth_token():
    token = environ.get("TVMTOOL_LOCAL_AUTHTOKEN")
    if token is None:
        raise AppException("TVM env variables wasn't set")
    return token


def _get_deploy_tvm_url():
    deploy_tvm_url = environ.get("DEPLOY_TVM_TOOL_URL")
    if deploy_tvm_url is None:
        raise AppException("TVM env variables wasn't set")
    return deploy_tvm_url
