from typing import Any, Dict
from tractor.disk.db import Database
from tractor.yandex_services.directory import Directory
from tractor.util.common import extract_login
from tractor_api.app.error_handling import AppException
from tractor_api.settings import settings
from tractor.csv.exceptions import CsvParserException
from tractor.csv.parser import CsvParser, decode_bytes_to_str
from tractor.disk.models import UserMigrationStatus, TaskType

from dataclasses import dataclass
from enum import Enum
from flask import request
import json


class SkipReason(str, Enum):
    NOT_FOUND_IN_YANDEX = "not_found_in_yandex"
    MIGRATION_ALREADY_IN_PROGRESS = "migration_already_in_progress"


@dataclass
class UserEntry:
    login: str


class CSVSizeTooLargeException(AppException):
    def __init__(self, limit: int) -> None:
        super(CSVSizeTooLargeException, self).__init__(
            error_code="csv_size_too_large",
            status_code=413,
            detail={"limit": limit},
        )


class LoginsCountTooLargeException(AppException):
    def __init__(self, limit: int) -> None:
        super(LoginsCountTooLargeException, self).__init__(
            error_code="logins_count_too_large",
            status_code=413,
            detail={"limit": limit},
        )


def _create_migration(
    org_id: str, provider: str, logins: list, env: Dict[str, Any]
) -> Dict[str, SkipReason]:
    yandex_directory: Directory = env["yandex_directory"]
    domain: str = yandex_directory.get_domain()
    yandex_users: list = yandex_directory.get_users()
    uid_mapping = _build_uid_mapping(yandex_users)

    skipped_logins: Dict[str, SkipReason] = {}
    db: Database = env["db"]
    with db.make_connection() as conn:
        with conn.cursor() as cur:
            # TODO lock users?
            for login in logins:
                formatted_login = extract_login(login)
                user = {
                    "uid": uid_mapping.get(formatted_login),
                    "login": formatted_login,
                    "email": "{}@{}".format(formatted_login, domain),
                }
                list_task_inp = json.dumps({"user": user, "provider": provider})
                migration = db.get_user_migration(org_id, formatted_login, cur)
                if migration is None:
                    list_task_id = db.create_task(
                        type=TaskType.LIST,
                        org_id=org_id,
                        domain=domain,
                        worker_input=list_task_inp,
                        cur=cur,
                    )
                    db.create_user_migration(org_id, domain, formatted_login, list_task_id, cur)
                else:
                    if (
                        migration.status != UserMigrationStatus.SUCCESS
                        and migration.status != UserMigrationStatus.ERROR
                    ):
                        skipped_logins[formatted_login] = SkipReason.MIGRATION_ALREADY_IN_PROGRESS
                        continue
                    list_task_id = db.create_task(
                        type=TaskType.LIST,
                        org_id=org_id,
                        domain=domain,
                        worker_input=list_task_inp,
                        cur=cur,
                    )
                    db.reset_user_migration(
                        migration.org_id,
                        migration.login,
                        migration.domain,
                        list_task_id,
                        cur,
                    )
    return skipped_logins


def _build_uid_mapping(yandex_users):
    ret = {}
    for yandex_user in yandex_users:
        login = extract_login(yandex_user["email"])
        ret[login] = yandex_user["uid"]
    return ret


def _get_logins_from_request_data():
    data = _check_limit_and_get_request_data()
    logins = _parse_request_data(data)
    _check_logins_limit(logins)
    return logins


def _check_limit_and_get_request_data() -> bytes:
    file_size_limit = settings().methods.disk.create_migration.file_size_limit
    # Check Content-Length header to prevent reading too large file.
    if request.content_length and request.content_length > file_size_limit:
        raise CSVSizeTooLargeException(file_size_limit)
    data = request.get_data(cache=False)
    if len(data) > file_size_limit:  # Potentialy Content-Length header could lie.
        raise CSVSizeTooLargeException(file_size_limit)
    return data


def _check_logins_limit(logins):
    logins_limit = settings().methods.disk.create_migration.logins_limit
    if len(logins) > logins_limit:
        raise LoginsCountTooLargeException(logins_limit)


def _csv_with_possible_delimiters_popped_back(data: str, delimiters=";,\t") -> str:
    split_data = data.splitlines()
    formatted_data = ""
    for line in split_data:
        formatted_data += line.rstrip(delimiters) + "\n"
    return formatted_data


def _parse_request_data(data: bytes):
    try:
        csv_string = decode_bytes_to_str(data)
        csv_string = _csv_with_possible_delimiters_popped_back(csv_string)
        user_data_source = CsvParser(csv_string, UserEntry)
        return [user_data.login for user_data in user_data_source]
    except CsvParserException as e:
        raise AppException(e.message, status_code=400, detail=e.params)


def _make_response(skipped_logins):
    logins = []
    for login, reason in skipped_logins.items():
        logins.append({"login": login, "reason": reason})
    return {"skipped_logins": logins}
