import dataclasses
from typing import Any, Dict, List
from tractor.disk.db import Database
from tractor.error import NOT_ENOUGH_QUOTA, YANDEX_USER_NOT_FOUND
from tractor.disk.models import (
    UserMigrationInfo,
    UserMigrationStatus,
    MigrationStats,
    OrgMigrationStatus,
)


def _fill_response_with_migration_statuses(org_id: str, env: Dict[str, Any], response: Dict):
    db: Database = env["db"]
    with db.make_connection() as conn:
        with conn.cursor() as cur:
            statuses = db.get_user_migration_statuses(org_id, cur)

    user_stats = _migration_stats(statuses)
    response["status"] = _migration_status(user_stats).value
    response["user_stats"] = dataclasses.asdict(user_stats)


def _fill_response_with_external_secret_status(org_id: str, env: Dict[str, Any], response: Dict):
    db: Database = env["db"]
    with db.make_connection() as conn:
        with conn.cursor() as cur:
            secret_status = db.get_external_secret_status_any_provider(org_id, cur)

    if secret_status.provider:
        response["provider"] = secret_status.provider
    response["external_secret_loaded"] = secret_status.external_secret_loaded


def _migration_stats(users_statuses: List[UserMigrationInfo]) -> MigrationStats:
    stats = MigrationStats()
    for user in users_statuses:
        if user.status == UserMigrationStatus.LISTING:
            stats.preparing += 1
        elif user.status == UserMigrationStatus.SYNCING:
            stats.in_progress += 1
        elif user.status == UserMigrationStatus.CANCELING:
            stats.canceling += 1
        elif user.status == UserMigrationStatus.ERROR:
            if user.error_reason == NOT_ENOUGH_QUOTA:
                stats.not_enough_quota += 1
            elif user.error_reason == YANDEX_USER_NOT_FOUND:
                stats.user_not_found += 1
            else:
                stats.error += 1
        elif user.status == UserMigrationStatus.SUCCESS:
            stats.success += 1
        else:
            raise RuntimeError("unknown user migration status: ", user.status)
    return stats


def _migration_status(stats: MigrationStats) -> OrgMigrationStatus:
    if stats.in_progress > 0 or stats.preparing > 0:
        return OrgMigrationStatus.IN_PROGRESS
    if stats.canceling > 0:
        return OrgMigrationStatus.CANCELING
    if stats.error > 0 or stats.user_not_found > 0 or stats.not_enough_quota > 0:
        return OrgMigrationStatus.ERROR
    if stats.success > 0:
        return OrgMigrationStatus.SUCCESS
    return OrgMigrationStatus.NOT_STARTED
