from typing import Dict, List
from flask import Response
from tractor.disk.db import Database
from tractor.yandex_services.directory import Directory
from tractor.disk.models import UserMigrationExportInfo
from tractor_api.settings import settings
from tractor_api.app import app
from tractor_api.app.responses import make_success_response
from tractor_api.app.requests import get_required_argument
from tractor_api.methods.disk.impl.get_migration import (
    _fill_response_with_migration_statuses,
    _fill_response_with_external_secret_status,
)
from tractor_api.methods.disk.impl.create_migration import (
    _create_migration,
    _get_logins_from_request_data,
    _make_response,
)
from tractor_api.app.tvm import user_tvm, allowed_service


@user_tvm.route("/organizations/<org_id>/migration", methods=["GET"])
@user_tvm.route("/organizations/<org_id>/migration/disk", methods=["GET"])
@allowed_service("sarah")
def get_migration(org_id: str) -> Response:
    response = {}
    env = _env_for_get()
    _fill_response_with_migration_statuses(org_id, env, response)
    _fill_response_with_external_secret_status(org_id, env, response)
    return make_success_response(data=response)


@user_tvm.route("/organizations/<org_id>/migration/export", methods=["GET"])
@user_tvm.route("/organizations/<org_id>/migration/disk/export", methods=["GET"])
@allowed_service("sarah")
def export_migration(org_id: str) -> Response:
    response: str = _export_migration(org_id, _env_for_export())
    return make_success_response(data=response, headers={"Content-Type": "text/csv"})


def _export_migration(org_id: str, env: Dict):
    db: Database = env["db"]
    with db.make_connection() as conn:
        with conn.cursor() as cur:
            export_migration: List[UserMigrationExportInfo] = db.get_user_migrations_for_export(
                org_id, cur
            )

    rows: List[str] = (len(export_migration) + 1) * [None]
    rows[
        0
    ] = "login,status,error_reason,yandex_disk_quota,external_disk_files_size,external_disk_files_count\n"
    for i, user in enumerate(export_migration):
        rows[
            i + 1
        ] = f'{user.login},{user.status.value},"{user.error_reason}",{_process_statistics_to_csv_string(user)}\n'
    return "".join(rows)


def _process_statistics_to_csv_string(user: UserMigrationExportInfo):
    val_or_empty = lambda field: field if field is not None else ""
    return (
        f"{val_or_empty(user.stats.quota)},{val_or_empty(user.stats.files_size)},"
        f"{val_or_empty(user.stats.files_count)}"
    )


@user_tvm.route("/organizations/<org_id>/migration", methods=["POST"])
@user_tvm.route("/organizations/<org_id>/migration/disk", methods=["POST"])
@allowed_service("sarah")
def create_migration(org_id: str) -> Response:
    provider = get_required_argument("provider")
    logins = _get_logins_from_request_data()
    skipped_logins = _create_migration(org_id, provider, logins, _env_for_create(org_id))
    response = _make_response(skipped_logins)
    if skipped_logins:
        return make_success_response(data=response, code=206)
    return make_success_response(data=response)


def _env_for_get():
    env = {"db": Database(settings().tractor_disk_db)}
    return env


def _env_for_export():
    env = {"db": Database(settings().tractor_disk_db)}
    return env


def _env_for_create(org_id):
    env = {
        "db": Database(settings().tractor_disk_db),
        "yandex_directory": Directory(org_id, settings().directory),
    }
    return env
