from typing import Any, Optional, Dict
from contextlib import closing
import json
from flask import Response
from tractor.versioned_keys import load as _load_versioned_keys
from tractor.disk.db import Database
from tractor.yandex_services.directory import Directory
from tractor.models import ExternalProvider
from tractor.crypto.fernet import Fernet
from tractor_api.settings import settings
from tractor_api.app.responses import make_error_response, make_success_response
from tractor_api.app.requests import get_required_argument, get_request_body_as_json
from tractor_api.app.tvm import service_tvm, user_tvm, allowed_service


@user_tvm.route("/organizations/<org_id>/external_secret", methods=["PUT"])
@allowed_service("sarah")
def put_external_secret(org_id: str) -> Response:
    provider_str: str = get_required_argument("provider")
    provider: ExternalProvider = ExternalProvider(provider_str)
    domain: str = _get_domain(org_id)
    # TODO: [RTEC-6938] Add run-time checking of `req_data` and `external_secret`:
    req_data: Dict[str, Any] = get_request_body_as_json()
    external_secret: str = req_data["external_secret"]
    db: Database = Database(settings().tractor_disk_db)
    fernet: Fernet = Fernet(_load_versioned_keys(settings()))
    _set_external_secret(org_id, domain, provider, external_secret, db, fernet)
    return make_success_response()


@service_tvm.route("/organizations/<org_id>/external_secret", methods=["GET"])
@allowed_service("collectors_ext")
def get_external_secret(org_id: str) -> Response:
    provider_str: str = get_required_argument("provider")
    provider: ExternalProvider = ExternalProvider(provider_str)
    domain: str = _get_domain(org_id)
    db: Database = Database(settings().tractor_disk_db)
    fernet: Fernet = Fernet(_load_versioned_keys(settings()))
    external_secret: Optional[str] = _get_external_secret(org_id, domain, provider, db, fernet)
    if external_secret is None:
        return make_error_response(404, "external_secret_not_found")
    data: Dict[str, Any] = {"external_secret": external_secret}
    return make_success_response(data=data)


def _get_domain(org_id: str) -> str:
    directory = Directory(org_id, settings().directory)
    return directory.get_domain()


def _set_external_secret(
    org_id: str,
    domain: str,
    provider: ExternalProvider,
    external_secret: str,
    db: Database,
    fernet: Fernet,
):
    with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
        encrypted_secret: bytes = fernet.encrypt_text(external_secret)
        db.set_external_secret(org_id, domain, provider, encrypted_secret, cur)


def _get_external_secret(
    org_id: str, domain: str, provider: ExternalProvider, db: Database, fernet: Fernet
) -> Optional[str]:
    with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
        encrypted_secret: Optional[bytes] = db.get_external_secret(
            org_id, domain, provider, cur=cur
        )
        if encrypted_secret is None:
            return None
        return fernet.decrypt_text(encrypted_secret)
