from datetime import datetime
from dataclasses import dataclass
from enum import Enum
import json
from base64 import b64encode
from flask import request
from typing import Any, Dict, Iterable, List, Optional
from tractor.crypto.fernet import Fernet
from tractor.csv.exceptions import CsvParserException
from tractor.csv.parser import CsvParser, decode_bytes_to_str
from tractor.mail.db import Database
from tractor.yandex_services.directory import Directory
from tractor.mail.models import (
    MailServerConnectionInfo,
    MailServerInfo,
    TaskType,
    UserMigrationStatus,
    UserInfo,
)
from tractor.util.common import extract_login
from tractor_api.app.error_handling import AppException
from tractor_api.app.requests import get_required_argument, get_required_arguments
from tractor_api.settings import settings


class SkipReason(str, Enum):
    MIGRATION_ALREADY_IN_PROGRESS = "migration_already_in_progress"
    WRONG_DATA_FORMAT_IN_CSV = "wrong_data_in_csv"


class CSVSizeTooLargeException(AppException):
    def __init__(self, limit: int) -> None:
        super(CSVSizeTooLargeException, self).__init__(
            error_code="csv_size_too_large",
            status_code=413,
            detail={"limit": limit},
        )


class UsersCountTooLargeException(AppException):
    def __init__(self, limit: int) -> None:
        super(UsersCountTooLargeException, self).__init__(
            error_code="logins_count_too_large",
            status_code=413,
            detail={"limit": limit},
        )


def get_mail_server_info() -> MailServerInfo:
    provider = get_required_argument("provider")
    if provider == "google" or provider == "microsoft":
        return MailServerInfo(provider, None)
    else:
        server, port, ssl = get_required_arguments("server", "port", "ssl")
        return MailServerInfo(provider, MailServerConnectionInfo(server, port, ssl))


def get_info_for_all_users_in_current_migration() -> List[UserInfo]:
    try:
        data: bytes = _check_limit_and_get_request_data()
        csv_string = decode_bytes_to_str(data)
        users_list = _convert_csv_to_users_info_list(csv_string)
        _check_logins_limit(users_list)
        return users_list
    except CsvParserException as e:
        raise AppException(e.message, status_code=400, detail=e.params)


def _convert_csv_to_users_info_list(csv_string) -> List[UserInfo]:
    user_data_source = CsvParser(csv_string, UserInfo)
    return [user_info for user_info in user_data_source]


def create_migration_impl(
    org_id: str,
    admin_uid: str,
    mail_server_info: MailServerInfo,
    all_users_info: List[UserInfo],
    env: Dict[str, Any],
) -> Dict[str, SkipReason]:
    yandex_directory: Directory = env["yandex_directory"]
    domain = yandex_directory.get_domain()
    skipped_logins: Dict[str, Any] = {}
    db: Database = env["db"]
    fernet: Fernet = env["fernet"]
    directory: Directory = env["yandex_directory"]

    valid_users_info = _get_valid_users_info(all_users_info)
    for user_info in valid_users_info:
        user_info.login = extract_login(user_info.login)

    for user in all_users_info:
        if user not in valid_users_info:
            skipped_logins[user.login] = SkipReason.WRONG_DATA_FORMAT_IN_CSV

    organization_users_list = directory.get_users()
    uid_mapping = _build_uid_mapping(organization_users_list)
    for user in valid_users_info:
        if uid_mapping.get(user.login) is None:
            continue
        user.uid = uid_mapping[user.login]

    with db.make_connection() as conn:
        with conn.cursor() as cur:
            for user_info in valid_users_info:
                prepare_task_input = _make_prepare_task_input(
                    user_info, mail_server_info, admin_uid, fernet
                )
                migration = db.get_user_migration(org_id, user_info.login, cur)
                if migration is None:
                    prepare_task_id = db.create_task(
                        type=TaskType.PREPARE,
                        org_id=org_id,
                        domain=domain,
                        worker_input=prepare_task_input,
                        cur=cur,
                    )

                    db.create_user_migration(org_id, domain, user_info.login, prepare_task_id, cur)
                else:
                    if (
                        migration.status != UserMigrationStatus.STOPPED
                        and migration.status != UserMigrationStatus.ERROR
                    ):
                        skipped_logins[user_info.login] = SkipReason.MIGRATION_ALREADY_IN_PROGRESS
                        continue
                    prepare_task_id = db.create_task(
                        type=TaskType.PREPARE,
                        org_id=org_id,
                        domain=domain,
                        worker_input=prepare_task_input,
                        cur=cur,
                    )
                    db.reset_user_migration(
                        migration.org_id,
                        migration.login,
                        migration.domain,
                        prepare_task_id,
                        cur,
                    )

    return skipped_logins


def _add_password_to_input_if_needed(
    user_info_dct: Dict[str, str], user_info: UserInfo, attr_name: str, fernet: Fernet
):
    password = getattr(user_info, attr_name)
    if password is not None:
        encrypted_password = b64encode(fernet.encrypt_text(password)).decode("ascii")
        user_info_dct["encrypted_" + attr_name] = encrypted_password


def _make_prepare_task_input(
    user_info: UserInfo, mail_server_info: MailServerInfo, admin_uid: str, fernet: Fernet
) -> str:
    user_info_dct = {
        "login": user_info.login,
        "first_name": user_info.first_name,
        "last_name": user_info.last_name,
        "middle_name": user_info.middle_name,
        "gender": user_info.gender,
        "birthday": user_info.birthday,
        "language": user_info.language,
        "uid": user_info.uid,
    }

    keys_with_none_values = []
    for key in user_info_dct.keys():
        if user_info_dct[key] is None:
            keys_with_none_values.append(key)

    _delete_given_keys(user_info_dct, keys_with_none_values)

    _add_password_to_input_if_needed(user_info_dct, user_info, "external_password", fernet)
    _add_password_to_input_if_needed(user_info_dct, user_info, "yandex_password", fernet)

    mail_server_info_dct = {"provider": mail_server_info.provider}
    if mail_server_info.provider == "custom":
        mail_server_info_dct.update(
            {
                "host": mail_server_info.conn_info.host,
                "port": mail_server_info.conn_info.port,
                "ssl": mail_server_info.conn_info.ssl,
            }
        )

    prepare_task_input = json.dumps(
        {
            "user": user_info_dct,
            "mail_server_info": mail_server_info_dct,
            "admin_uid": admin_uid,
        }
    )
    return prepare_task_input


def _check_limit_and_get_request_data() -> bytes:
    file_size_limit = settings().methods.mail.create_migration.file_size_limit
    # Check Content-Length header to prevent reading too large file.
    if request.content_length and request.content_length > file_size_limit:
        raise CSVSizeTooLargeException(file_size_limit)
    data = request.get_data(cache=False)
    if len(data) > file_size_limit:  # Potentially Content-Length header could lie.
        raise CSVSizeTooLargeException(file_size_limit)
    return data


def _validate_user_info(user_info: UserInfo) -> bool:
    if (user_info.gender is not None and user_info.gender not in Directory.ALLOWED_GENDERS) or (
        user_info.language is not None and user_info.language not in Directory.ALLOWED_LANGUAGES
    ):
        return False

    try:
        if user_info.birthday is not None:
            datetime.strptime(user_info.birthday, Directory.USER_BIRTHDAY_FORMAT)
        return True
    except:
        return False


def _get_valid_users_info(users: List[UserInfo]) -> List[UserInfo]:
    result = list(filter(_validate_user_info, users))
    return result


def _check_logins_limit(users):
    logins_limit = settings().methods.mail.create_migration.logins_limit
    if len(users) > logins_limit:
        raise UsersCountTooLargeException(logins_limit)


def _delete_given_keys(instance: type, keys: Iterable):
    for key in keys:
        del instance[key]


def _build_uid_mapping(yandex_users) -> Dict[str, str]:
    ret = {}
    for yandex_user in yandex_users:
        login = extract_login(yandex_user["email"])
        ret[login] = yandex_user["uid"]
    return ret


def make_response(skipped_logins: Dict[str, SkipReason]):
    logins = []
    for login, reason in skipped_logins.items():
        logins.append({"login": login, "reason": reason})
    return {"skipped_logins": logins}
