from typing import List, Dict, Any
from tractor.mail.db import Database

from tractor.mail.models import (
    UserMigrationInfo,
    UserMigrationStatus,
    MigrationStats,
    OrgMigrationStatus,
)
from tractor.models import ExternalSecretStatus


def get_migration_stats(org_id: str, env: Dict[str, Any]):
    db: Database = env["db"]
    with db.make_connection() as conn:
        with conn.cursor() as cur:
            migration_infos = db.get_user_migration_statuses(org_id, cur)
            return _aggregate_migration_stats(migration_infos)


def _aggregate_migration_stats(infos: List[UserMigrationInfo]) -> MigrationStats:
    stats = MigrationStats()
    for info in infos:
        if info.status == UserMigrationStatus.PREPARING:
            stats.preparing += 1
        elif info.status == UserMigrationStatus.INITIAL_SYNC:
            stats.initial_sync += 1
        elif info.status == UserMigrationStatus.SYNC_NEWEST:
            stats.sync_newest += 1
        elif info.status == UserMigrationStatus.ERROR:
            stats.error += 1
        elif info.status == UserMigrationStatus.STOPPING:
            stats.stopping += 1
        elif info.status == UserMigrationStatus.STOPPED:
            stats.success += 1
        else:
            raise RuntimeError("Unknown user migration status: ", info.status)
    return stats


def calculate_migration_status(stats: MigrationStats) -> OrgMigrationStatus:
    if stats.preparing > 0 or stats.initial_sync > 0 or stats.sync_newest > 0:
        return OrgMigrationStatus.IN_PROGRESS
    if stats.stopping > 0:
        return OrgMigrationStatus.STOPPING
    if stats.error > 0:
        return OrgMigrationStatus.ERROR
    if stats.success > 0:
        return OrgMigrationStatus.SUCCESS
    return OrgMigrationStatus.NOT_STARTED


def get_external_secret_status(org_id: str, env: Dict[str, Any]):
    db: Database = env["db"]
    with db.make_connection() as conn:
        with conn.cursor() as cur:
            return db.get_external_secret_status_any_provider(org_id, cur)


def make_response(
    stats: MigrationStats, status: OrgMigrationStatus, secret_status: ExternalSecretStatus
):
    response = {
        "status": status.value,
        "user_stats": {
            "preparing": stats.preparing,
            "initial_sync": stats.initial_sync,
            "sync_newest": stats.sync_newest,
            "stopping": stats.stopping,
            "success": stats.success,
            "error": stats.error,
        },
        "external_secret_loaded": secret_status.external_secret_loaded,
    }
    _fill_provider(status, secret_status, response)
    return response


def _fill_provider(
    status: OrgMigrationStatus, secret_status: ExternalSecretStatus, response: Dict[str, Any]
):
    if secret_status.external_secret_loaded:
        response["provider"] = secret_status.provider
    elif status != OrgMigrationStatus.NOT_STARTED:
        response["provider"] = "custom"
