from typing import Any, List, Dict
from flask import Response, g
from tractor.crypto.fernet import Fernet
from tractor.versioned_keys import load as load_versioned_keys
from tractor.mail.models import UserMigrationExportInfo
from tractor.mail.db import Database
from tractor.yandex_services.directory import Directory
from tractor_api.methods.mail.impl.create_migration import (
    SkipReason,
    get_info_for_all_users_in_current_migration,
    get_mail_server_info,
    create_migration_impl,
    make_response as make_response_for_create,
)
from tractor_api.methods.mail.impl.get_migration import (
    get_migration_stats,
    calculate_migration_status,
    make_response as make_response_for_get,
    get_external_secret_status,
)
from tractor_api.methods.mail.impl.export_migration import export_user_migrations, render_export_csv
from tractor_api.methods.mail.impl.stop_migration import stop_migration_impl
from tractor_api.settings import settings
from tractor_api.app.responses import make_success_response
from tractor_api.app import app
from tractor_api.app.tvm import user_tvm, allowed_service


@user_tvm.route("/organizations/<org_id>/migration/mail", methods=["POST"])
@allowed_service("sarah")
def create_mail_migration(org_id: str) -> Response:
    mail_server_info = get_mail_server_info()
    users_info = get_info_for_all_users_in_current_migration()
    skipped_logins = create_migration_impl(
        org_id, g.uid, mail_server_info, users_info, _env_for_create(org_id)
    )
    response = make_response_for_create(skipped_logins)
    if skipped_logins:
        return make_success_response(data=response, code=206)
    return make_success_response(data=response)


@user_tvm.route("/organizations/<org_id>/migration/mail", methods=["GET"])
@allowed_service("sarah")
def get_mail_migration(org_id: str) -> Response:
    env = _env_for_get()
    stats = get_migration_stats(org_id, env)
    migration_status = calculate_migration_status(stats)
    secret_status = get_external_secret_status(org_id, env)
    response = make_response_for_get(stats, migration_status, secret_status)
    return make_success_response(data=response)


@user_tvm.route("/organizations/<org_id>/migration/mail/export", methods=["GET"])
@allowed_service("sarah")
def export_mail_migration(org_id: str) -> Response:
    db: Database = Database(settings().tractor_disk_db)
    stats: List[UserMigrationExportInfo] = export_user_migrations(org_id, db=db)
    csv: str = render_export_csv(stats)
    return make_success_response(data=csv, headers={"Content-Type": "text/csv"})


@user_tvm.route("/organizations/<org_id>/migration/mail/stop", methods=["POST"])
@allowed_service("sarah")
def stop_mail_migration(org_id: str) -> Response:
    env = _env_for_stop(org_id)
    stop_migration_impl(org_id, env)
    return make_success_response()


def _env_for_create(org_id: str) -> Dict[str, Any]:
    env = {
        "db": Database(settings().tractor_disk_db),
        "yandex_directory": Directory(org_id, settings().directory),
        "fernet": Fernet(load_versioned_keys(settings())),
    }
    return env


def _env_for_get() -> Dict[str, Any]:
    env = {
        "db": Database(settings().tractor_disk_db),
    }
    return env


def _env_for_stop(org_id: str) -> Dict[str, Any]:
    env = {
        "db": Database(settings().tractor_disk_db),
        "directory": Directory(org_id, settings().directory),
    }
    return env
