from typing import Any, Callable, Generator
from inspect import isgenerator
from functools import wraps

from tractor.exception import TemporaryError


def get_source_name(source):
    if isinstance(source, str):
        return source
    else:
        return type(source).__name__


class DiskError(Exception):
    def __init__(self, message, source):
        self.name = type(self).__name__
        self.message = message
        self.source = get_source_name(source)

    def __repr__(self):
        return '<{}: {}. Details: "{}">'.format(self.source, self.name, self.message)

    __str__ = __repr__


class DiskTemporaryError(DiskError, TemporaryError):
    NAME = "temporary"


class DiskPermanentError(DiskError):
    NAME = "permanent"


class ExternalUserNotFound(DiskPermanentError):
    def __init__(self, source):
        super().__init__("User not found in external disk", source)


def exception_to_category(exception):
    err_category = None
    try:
        raise exception
    except (DiskTemporaryError, DiskPermanentError) as disk_err:
        exception = disk_err
        err_category = "{}_{}".format(disk_err.source.lower(), disk_err.NAME)
    except Exception as e:
        exception = e
        err_category = "internal"
    return exception, err_category


def catch_errors(method, dispatch_error: Callable[[Exception], Any]):
    def wrap_generator(generator: Generator):
        try:
            yield from generator
        except Exception as exc:
            dispatch_error(exc)
            return

    @wraps(method)
    def impl(*args, **kwargs):
        try:
            res = method(*args, **kwargs)
        except Exception as exc:
            dispatch_error(exc)
            return
        if isgenerator(res):
            return wrap_generator(res)
        return res

    return impl
