from hashlib import sha1
from collections import defaultdict
from os.path import splitext
from tractor_disk.common import LOST_AND_FOUND_DIR, SHARED_WITH_ME_DIR, replace_slash

from google.oauth2 import service_account
from google.auth.exceptions import GoogleAuthError
import googleapiclient.discovery
from googleapiclient.http import MediaIoBaseDownload, HttpRequest
from googleapiclient.errors import HttpError
import io
from dataclasses import dataclass
from typing import Any, Dict
from tractor_disk.disk_error import (
    DiskTemporaryError,
    DiskPermanentError,
    ExternalUserNotFound,
    catch_errors,
)
from tractor.util.retrying import need_retry, TemporaryError
from tractor_disk.settings import settings

SCOPES = ["https://www.googleapis.com/auth/drive"]
FOLDER_MIME_TYPE = "application/vnd.google-apps.folder"
FIELDS_LISTING = ",".join(
    [
        "nextPageToken",
        "files/parents",
        "files/id",
        "files/name",
        "files/mimeType",
        "files/ownedByMe",
        "files/size",
        "files/md5Checksum",
        "files/shared",
        "files/exportLinks",
    ]
)
USER_NOT_FOUND_MSG = "invalid_grant: Invalid email or User ID"


@dataclass
class GoogleDocExportFormat:
    mimetype: str
    file_extension: str


GOOGLE_DOC_FORMATS = {
    "application/vnd.google-apps.document": GoogleDocExportFormat(
        "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
        ".docx",
    ),
    "application/vnd.google-apps.drawing": GoogleDocExportFormat(
        "image/svg+xml",
        ".svg",
    ),
    "application/vnd.google-apps.presentation": GoogleDocExportFormat(
        "application/vnd.openxmlformats-officedocument.presentationml.presentation",
        ".pptx",
    ),
    "application/vnd.google-apps.script": GoogleDocExportFormat(
        "application/vnd.google-apps.script+json",
        "",
    ),
    "application/vnd.google-apps.spreadsheet": GoogleDocExportFormat(
        "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        ".xlsx",
    ),
}


DISK_NAME = "GoogleDrive"


def catch_disk_errors(method):
    return catch_errors(method, _dispatch_disk_error)


def _dispatch_disk_error(exc: Exception):
    try:
        raise exc
    except (DiskTemporaryError, DiskPermanentError) as disk_err:
        if not disk_err.source:
            disk_err.source = DISK_NAME
        raise disk_err
    except GoogleAuthError as auth_err:
        msg = _get_message_from_google_error(auth_err)
        if msg == USER_NOT_FOUND_MSG:
            raise ExternalUserNotFound(DISK_NAME)
        raise DiskTemporaryError(msg if msg else "Auth error", DISK_NAME)
    except HttpError as http_err:
        msg = str(http_err)
        if need_retry(http_err.status_code):
            raise DiskTemporaryError(msg, DISK_NAME)
        else:
            raise DiskPermanentError(msg, DISK_NAME)
    except (TemporaryError, TimeoutError, ConnectionError, OSError) as tmp_err:
        raise DiskTemporaryError(str(tmp_err), DISK_NAME)


def _get_message_from_google_error(err):
    if type(err.args) == str:
        return err.args
    if type(err.args) == tuple and len(err.args) > 0 and type(err.args[0]) == str:
        return err.args[0]
    return None


class GoogleDrive:
    def __init__(self, secret, user_email, chunk_size=10 * 1024 * 1024):
        credentials = service_account.Credentials.from_service_account_info(secret, scopes=SCOPES)
        delegated_credentials = credentials.with_subject(user_email)
        self.gdrive = googleapiclient.discovery.build(
            "drive", "v3", credentials=delegated_credentials
        )
        self.chunk_size = chunk_size
        self.root_id = None
        self.settings = settings().google_drive

    @catch_disk_errors
    def root_folder_id(self):
        if self.root_id:
            return self.root_id
        response = (
            self.gdrive.files().get(fileId="root").execute(num_retries=self.settings.retrying.count)
        )
        self.root_id = response.get("id")
        if not self.root_id:
            raise RuntimeError("Cannot get root folder id")
        return self.root_id

    @catch_disk_errors
    def get_files(self):
        ret = []
        page_token = None
        while True:
            response = (
                self.gdrive.files()
                .list(pageToken=page_token, fields=FIELDS_LISTING)
                .execute(num_retries=self.settings.retrying.count)
            )
            ret += response.get("files", [])
            page_token = response.get("nextPageToken")
            if not page_token:
                break
        return ret

    @staticmethod
    def is_downloadable(file_metadata: Dict[str, Any]) -> bool:
        if file_metadata.get("exportLinks", {}):
            return True
        mimeType: str = file_metadata["mimeType"]
        if mimeType.startswith("application/vnd.google-apps."):
            # Assume that such files, if anything, can only be exported.
            return mimeType in GOOGLE_DOC_FORMATS
        return True  # Assume that non-`google-apps` files can be downloaded directly.

    @catch_disk_errors
    def download_file(self, file_id):
        res = (
            self.gdrive.files()
            .get(fileId=file_id, fields="exportLinks,mimeType")
            .execute(num_retries=self.settings.retrying.count)
        )
        if "exportLinks" in res:
            return self._download_doc(file_id, res.get("exportLinks"), res.get("mimeType"))
        else:
            return self._download_binary(file_id)

    def folder_mime_type(self):
        return FOLDER_MIME_TYPE

    @catch_disk_errors
    def _download_binary(self, file_id):
        request = self.gdrive.files().get_media(fileId=file_id)
        return self._download(request)

    @catch_disk_errors
    def _download_doc(self, file_id, export_links, mime_type):
        export_mime_type, download_uri = self._export_data(mime_type, export_links)
        request = self.gdrive.files().export_media(fileId=file_id, mimeType=export_mime_type)
        content = self._download(request)
        try:
            for chunk in content:
                yield chunk
        except HttpError as e:
            if _export_size_limit_exceeded(e):
                request = HttpRequest(self.gdrive._http, lambda *args: None, download_uri)
                yield from self._download(request)
            else:
                raise

    @catch_disk_errors
    def _download(self, request):
        content = io.BytesIO()
        downloader = MediaIoBaseDownload(content, request, chunksize=self.chunk_size)
        done = False
        while not done:
            status, done = downloader.next_chunk(num_retries=self.settings.retrying.count)
            chunk = content.getvalue()
            # print ("Download {}%, chunk size={}".format(int(status.progress() * 100), len(chunk)))
            yield chunk
            content.truncate(0)
            content.seek(0)

    def _export_data(self, mime_type, export_links):
        fmt = GOOGLE_DOC_FORMATS.get(mime_type)
        if fmt:
            return fmt.mimetype, export_links[fmt.mimetype]
        else:
            return export_links.popitem()


class GoogleBuildPathMappingOp:
    def __init__(self, files):
        self.files = files
        self.file_by_id_mapping = {}
        self.path_by_id_mapping = {}
        self.multi_parent_files = set()

    def __call__(self):
        self.file_by_id_mapping = self._build_file_by_id_mapping(self.files)
        file_by_path_mapping = self._build_file_by_path_mapping()
        self._make_paths_unique(file_by_path_mapping)
        return self.path_by_id_mapping, list(self.multi_parent_files)

    def _build_file_by_id_mapping(self, files):
        file_by_id = {}
        for file in files:
            file_by_id[file["id"]] = file
        return file_by_id

    def _fill_path_for_file(self, file):
        if file["id"] in self.path_by_id_mapping:
            return self.path_by_id_mapping[file["id"]]
        if file["id"] == file["root_folder_id"]:
            raise RuntimeError("Path for root not defined")

        name = replace_slash(file["name"])
        if self._parent_is_root(file):
            return self._cache_and_get_decorated_path(file, name)
        if not self._has_parent(file):
            fake_parent_path = SHARED_WITH_ME_DIR if file["shared"] else LOST_AND_FOUND_DIR
            path = "{}/{}".format(fake_parent_path, name)
            return self._cache_and_get_decorated_path(file, path)
        parent_id = self._file_parent_id(file)
        parent = self.file_by_id_mapping[parent_id]
        parent_path = self._fill_path_for_file(parent)
        path = "{}/{}".format(parent_path, name)
        return self._cache_and_get_decorated_path(file, path)

    def _cache_and_get_decorated_path(self, file, path):
        self.path_by_id_mapping[file["id"]] = _decorate_path(path, file["mimeType"])
        return self.path_by_id_mapping[file["id"]]

    def _parent_is_root(self, file):
        if not "parents" in file or len(file["parents"]) == 0:
            return False
        parent_id = self._file_parent_id(file)
        return parent_id == file["root_folder_id"]

    def _has_parent(self, file):
        if "parents" not in file:
            return False
        if len(file["parents"]) == 0:
            return False
        parent_id = file["parents"][0]
        if parent_id not in self.file_by_id_mapping:
            return False
        return True

    def _file_parent_id(self, file):
        if "parents" not in file:
            raise RuntimeError('No "parents" field for file_id={}'.format(file["id"]))
        if len(file["parents"]) == 0:
            raise RuntimeError("Empty parents for file_id={}".format(file["id"]))
        if len(file["parents"]) > 1:
            self.multi_parent_files.add(file["id"])
        return file["parents"][0]

    def _build_file_by_path_mapping(self):
        file_by_path_mapping = defaultdict(list)
        for file in self.files:
            full_path = self._fill_path_for_file(file)
            file_by_path_mapping[full_path].append(file)
        return file_by_path_mapping

    def _make_paths_unique(self, file_by_path_mapping: defaultdict(list)):
        for file_path, files in file_by_path_mapping.items():
            if len(files) > 1:
                for file in files:
                    self.path_by_id_mapping[file["id"]] = _append_hash_of_id(file["id"], file_path)


def _decorate_path(file_path, mimetype):
    if mimetype not in GOOGLE_DOC_FORMATS:
        return file_path
    fmt = GOOGLE_DOC_FORMATS[mimetype]
    if file_path.endswith(fmt.file_extension):
        return file_path
    return file_path + fmt.file_extension


def _append_hash_of_id(file_id, file_path):
    hasher = sha1()
    hasher.update(file_id.encode())
    hash = hasher.hexdigest()[0:10]
    path, ext = splitext(file_path)
    file_path = f"{path} ({hash}){ext}"

    return file_path


def _export_size_limit_exceeded(http_error):
    try:
        details = http_error.error_details
        return details[0]["reason"] == "exportSizeLimitExceeded"
    except:
        return False
