#!/usr/bin/env python3

import time
import code, traceback, signal

from tractor.logger import setup_logger
from tractor_disk.settings import settings
from tractor_disk.workers.env import Env, make_env, make_list_task_env as make_task_env
from tractor_disk.workers.list_worker import run_task, acquire_task
from tractor_disk.workers.heartbeat import heartbeat


def main():
    global_env = make_env()
    settings = global_env["settings"]
    while True:
        task = try_acquire_task(global_env)
        if task:
            # Heartbeating should start as soon as possible after acquiring a task.
            with heartbeat(global_env, task.task_id):
                global_env["logger"].info(message="acquired listing task", task_id=task.task_id)
                process_task(task, global_env)
        else:
            global_env["logger"].info(message="no listing tasks acquired")
            time.sleep(settings.list.worker_sleep_period_in_seconds)


def try_acquire_task(global_env: Env):
    logger = global_env["logger"]
    try:
        return acquire_task(global_env)
    except Exception as e:
        logger.exception("failed to acquire list task", exception=e)


def process_task(task, global_env: Env):
    logger = global_env["logger"]
    try:
        task_env = make_task_env(task, global_env)
        if task_env:
            run_task(task, task_env)
    except Exception as e:
        logger.exception(
            "failed to process list task",
            exception=e,
        )


def debug(sig, frame):
    """Interrupt running process, and provide a python prompt for
    interactive debugging."""
    d = {"_frame": frame}  # Allow access to frame object.
    d.update(frame.f_globals)  # Unless shadowed by global
    d.update(frame.f_locals)

    i = code.InteractiveConsole(d)
    message = "Signal received : entering python shell.\nTraceback:\n"
    message += "".join(traceback.format_stack(frame))
    i.interact(message)


def listen():
    signal.signal(signal.SIGUSR1, debug)  # Register handler


if __name__ == "__main__":
    listen()
    setup_logger(settings().logging)
    main()
